<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class Employee extends Model
{
protected $fillable = [
'user_id',
'name',
'dob',
'gender',
'phone',
'address',
'email',
'password',
'employee_id',
'branch_id',
'department_id',
'designation_id',
'company_doj',
'documents',
'account_holder_name',
'account_number',
'bank_name',
'bank_identifier_code',
'branch_location',
'tax_payer_id',
'salary_type',
'biometric_emp_id',
'account',
'salary',
'created_by',
'reporting_manager',
'team_lead',
'role',
'uan_number',
'pf_number',
'esi_number',
'pran_number',
'aadhar_number',
'pan_number',

];
protected static function boot()
{
    parent::boot();

    static::creating(function ($employee) {
        if (empty($employee->employee_id)) {
            $maxId = Employee::max(DB::raw('CAST(employee_id AS UNSIGNED)'));
            $employee->employee_id = $maxId ? $maxId + 1 : 1;
        }
    });
}

public function documents()
{
return $this->hasMany('App\Models\EmployeeDocument', 'employee_id', 'employee_id')->get();
}

public function salary_type()
{
return $this->hasOne('App\Models\PayslipType', 'id', 'salary_type')->pluck('name')->first();
}

public function allowances()
{
return $this->hasMany(Allowance::class);
}

public function commissions()
{
return $this->hasMany(Commission::class);
}

public function loans()
{
return $this->hasMany(Loan::class);
}

public function saturationDeductions()
{
return $this->hasMany(SaturationDeduction::class);
}

public function otherPayments()
{
return $this->hasMany(OtherPayment::class);
}

public function roleUser()
{

return $this->belongsTo(User::class, 'role', 'type');
}

public function overtimes()
{
return $this->hasMany(Overtime::class);
}

public function get_net_salary()
{

$total_actual_allowance = $this->allowances->sum('actual_amount');

$total_commission = $this->commissions->sum('amount');

$total_other_payment = $this->otherPayments->sum('amount');

$total_overtime = $this->overtimes->reduce(function ($carry, $ot) {
return $carry + ($ot->number_of_days ?? 0) * ($ot->hours ?? 0) * ($ot->rate ?? 0);
}, 0);

$total_actual_earning = $total_actual_allowance + $total_commission + $total_other_payment + $total_overtime;

// Calculate total allowances
$total_allowance = $this->allowances->sum(function ($allowance) {
return ($allowance->type === 'percent') ? ($allowance->amount * $this->salary / 100) : $allowance->amount ;
});

// Calculate total commissions
$total_commission = $this->commissions->sum(function ($commission) {
return ($commission->type === 'percent') ? ($commission->amount * $this->salary / 100): $commission->amount ;
});

// Calculate total loans
$total_loan = $this->loans->sum(function ($loan) {
return ($loan->type === 'percent') ? ($loan->amount * $this->salary / 100):$loan->amount  ;
});

// Calculate total saturation deductions
$total_saturation_deduction = $this->saturationDeductions->sum(function ($deduction) {
return ($deduction->type === 'percent') ? ($deduction->amount * $this->salary / 100) : $deduction->amount ;
});

// Calculate total other payments
$total_other_payment = $this->otherPayments->sum(function ($otherPayment) {
return ($otherPayment->type === 'percent') ?  ($otherPayment->amount * $this->salary / 100): $otherPayment->amount ;
});

// Calculate total overtime
$total_over_time = $this->overtimes->sum(function ($over_time) {
return $over_time->number_of_days * $over_time->hours * $over_time->rate;
});

// Calculate net salary
$total_earning =   $total_allowance + $total_commission + $total_other_payment + $total_over_time;

$total_deduction = $total_loan + $total_saturation_deduction;

$net_salary = $total_actual_earning - $total_deduction;
$net_salary = max(0, $net_salary); // Returns 0 if net_salary is negative
return sprintf("%.2f", $net_salary); // Formats to 2 decimal places, e.g., 0.00
}


public static function allowance($id)
{
$allowances  = Allowance::where('employee_id', '=', $id)->get();

$total_allowance = 0;
foreach($allowances as $allowance)
{
$total_allowance += $allowance->amount;
}

$allowance_json = json_encode($allowances);  // ← JSON CREATED HERE

return $allowance_json;
}
public static function commission($id)
{
//commission
$commissions  = Commission::where('employee_id', '=', $id)->get();
$total_commission = 0;
foreach($commissions as $commission)
{
$total_commission = $commission->amount + $total_commission;
}
$commission_json = json_encode($commissions);

return $commission_json;

}

public static function loan($id)
{
//Loan
$loans  = Loan::where('employee_id', '=', $id)->get();
$total_loan = 0;
foreach($loans as $loan)
{
$total_loan = $loan->amount + $total_loan;
}
$loan_json = json_encode($loans);

return $loan_json;
}

public static function saturation_deduction($id)
{
//Saturation Deduction
$saturation_deductions  = SaturationDeduction::where('employee_id', '=', $id)->get();
$total_saturation_deduction = 0;
foreach($saturation_deductions as $saturation_deduction)
{
$total_saturation_deduction = $saturation_deduction->amount + $total_saturation_deduction;
}
$saturation_deduction_json = json_encode($saturation_deductions);

return $saturation_deduction_json;

}

public static function other_payment($id)
{
//OtherPayment
$other_payments  = OtherPayment::where('employee_id', '=', $id)->get();
$total_other_payment = 0;
foreach($other_payments as $other_payment)
{
$total_other_payment = $other_payment->amount + $total_other_payment;
}
$other_payment_json = json_encode($other_payments);

return $other_payment_json;
}

public static function overtime($id)
{
//Overtime
$over_times  = Overtime::where('employee_id', '=', $id)->get();
$total_over_time = 0;
foreach($over_times as $over_time)
{
$total_work  = $over_time->number_of_days * $over_time->hours;
$amount  = $total_work * $over_time->rate;
$total_over_time = $amount + $total_over_time;
}
$over_time_json = json_encode($over_times);

return $over_time_json;
}

public static function employee_id()
{
$employee = Employee::latest()->first();

return !empty($employee) ? $employee->id + 1 : 1;
}


public function reportingManager()
{
return $this->belongsTo(User::class, 'reporting_manager');
}

public function teamLead()
{
return $this->belongsTo(User::class, 'team_lead');
}


public function branch()
{
return $this->hasOne('App\Models\Branch', 'id', 'branch_id');
}

public function department()
{
return $this->hasOne('App\Models\Department', 'id', 'department_id');
}

public function designation()
{
return $this->hasOne('App\Models\Designation', 'id', 'designation_id');
}

public function salaryType()
{
return $this->hasOne('App\Models\PayslipType', 'id', 'salary_type');
}

public function user()
{
return $this->hasOne('App\Models\User', 'id', 'user_id');
}

public function paySlip()
{
return $this->hasOne('App\Models\PaySlip', 'id', 'employee_id');
}

public function bankAccount()
{
return $this->hasOne('App\Models\BankAccount', 'id', 'account');
}


public function present_status($employee_id, $data)
{
return AttendanceEmployee::where('employee_id', $employee_id)->where('date', $data)->first();
}


public static function employee_salary($salary)
{
$employee = Employee::where("salary", $salary)->first();
if ($employee->salary == '0' || $employee->salary == '0.0') {
return "-";
} else {
return $employee->salary;
}
}

public function generateEmployeeIdAjax(Request $request)
{
    $prefix = $request->prefix ?? Utility::getValByName('employee_prefix') ?? '#EMP';
    $nextId = Utility::getNextEmployeeId($prefix);
    return response()->json($nextId);
}

}
