<?php

namespace App\Http\Controllers;

use App\Exports\InvoiceExport;
use App\Models\BankAccount;
use App\Models\CreditNote;
use App\Models\Customer;
use App\Models\CustomField;
use App\Models\Invoice;
use App\Models\InvoiceBankTransfer;
use App\Models\InvoicePayment;
use App\Models\InvoiceProduct;
use App\Models\Plan;
use App\Models\ProductService;
use App\Models\ProductServiceCategory;
use App\Models\StockReport;
use App\Models\Transaction;
use App\Models\Utility;
use App\Models\TransactionLines;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\EWayBill;
use App\Models\EWayBillProduct;

class InvoiceController extends Controller
{
    public function __construct()
    {
    }

    public function index(Request $request)
    {
        if (\Auth::user()->can('manage invoice')) {
            $customer = Customer::where('created_by', '=', \Auth::user()->creatorId())->get()->pluck('name', 'id');
            $customer->prepend('Select Customer', '');
            $status = Invoice::$statues;
            $query = Invoice::where('created_by', '=', \Auth::user()->creatorId());

            if (!empty($request->customer)) {
                $query->where('customer_id', '=', $request->customer);
            }
            if (count(explode('to', $request->issue_date)) > 1) {
                $date_range = explode(' to ', $request->issue_date);
                $query->whereBetween('issue_date', $date_range);
            } elseif (!empty($request->issue_date)) {
                $date_range = [$request->issue_date, $request->issue_date];
                $query->whereBetween('issue_date', $date_range);
            }
            if ($request->status != null) {
                $query->where('status', '=', $request->status);
            }
            $invoices = $query->get();

            return view('invoice.index', compact('invoices', 'customer', 'status'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }

    public function create($customerId = 0)
    {
        if (\Auth::user()->can('create invoice')) {
            // Fetch custom fields for invoice
            $customFields = CustomField::where('created_by', \Auth::user()->creatorId())
                ->where('module', 'invoice')
                ->get();
    
            // Generate invoice number
            $invoice_number = \Auth::user()->invoiceNumberFormat($this->invoiceNumber());
    
            // Fetch customers
            $customers = Customer::where('created_by', \Auth::user()->creatorId())
                ->get()
                ->pluck('name', 'id');
            $customers->prepend('Select Customer', '');
    
            // Fetch product/service categories (income)
            $category = ProductServiceCategory::where('created_by', \Auth::user()->creatorId())
                ->where('type', 'income')
                ->get()
                ->pluck('name', 'id');
            $category->prepend('Select Category', '');
    
            // Fetch products/services
            $product_services = ProductService::where('created_by', \Auth::user()->creatorId())
                ->get()
                ->pluck('name', 'id');
            $product_services->prepend('--', '');
    
            // Fetch customer if $customerId is provided
            $customer = $customerId ? Customer::find($customerId) : null;
    
            // Fetch company settings
            $settings_data = Utility::settingsById(\Auth::user()->creatorId());
            $company_state = $settings_data['company_state'] ?? '';
            $billing_state = $customer ? strtolower(trim($customer->billing_state ?? '')) : '';
    
            // Determine tax type (VAT or GST)
            $tax_type = strtolower($settings_data['tax_type'] ?? 'gst'); // fallback to GST if not set
    
            return view('invoice.create', compact(
                'customers',
                'invoice_number',
                'product_services',
                'category',
                'customFields',
                'customerId',
                'customer',
                'settings_data',
                'company_state',
                'billing_state',
                'tax_type' // Pass tax type to view
            ));
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }
    

    public function customer(Request $request)
    {
        $customer = Customer::where('id', '=', $request->id)->first();

        if ($customer) {
            $billingState = strtolower(trim($customer->billing_state ?? ''));
            $companyState = strtolower(trim(\Auth::user()->settings_data['company_state'] ?? 'defaultstate'));

            $customerDetailView = view('invoice.customer_detail', compact('customer'))->render();

            return response()->json([
                'view' => $customerDetailView,
                'state' => $billingState,
                'city' => $customer->billing_city,
                'country' => $customer->billing_country,
                'address' => $customer->billing_address,
                'postal_code' => $customer->billing_zip,
            ]);
        }

        return response()->json(['error' => 'Customer not found'], 404);
    }

    public function product(Request $request)
    {
        $data['product'] = $product = ProductService::find($request->product_id);
        if (!$product) {
        
            return json_encode(['error' => 'Product not found']);
        }
        $data['unit'] = (!empty($product->unit)) ? $product->unit->name : '';
        $data['tax_id'] = $product->tax_id ?? 0;
        $data['taxes'] = $product->tax_id ? $product->tax($product->tax_id) : [];
    
        $data['salePrice'] = floatval($product->sale_price);
        $data['quantity'] = 1;
        $data['discount'] = 0;
        $data['totalAmount'] = $data['salePrice'] - $data['discount'];

        return json_encode($data);
    }
    public function store(Request $request)
    {
        if (!\Auth::user()->can('create invoice')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    
        try {
            // Validate input
            $validator = \Validator::make(
                $request->all(),
                [
                    'customer_id' => 'required',
                    'issue_date' => 'required|date',
                    'due_date' => 'required|date|after_or_equal:issue_date',
                    'category_id' => 'required',
                    'items' => 'required|array',
                    'items.*.item' => 'required|exists:product_services,id',
                    'items.*.quantity' => 'required|numeric|min:0',
                    'items.*.price' => 'required|numeric|min:0',
                    'items.*.discount' => 'required|numeric|min:0',
                ]
            );
    
            if ($validator->fails()) {
                return redirect()->back()->with('error', $validator->getMessageBag()->first());
            }
    
            // Fetch tax type from settings
            $settings = Utility::settingsById(\Auth::user()->creatorId());
            $tax_type = strtolower($settings['tax_type'] ?? 'gst'); // fallback to GST
    
            // Create new invoice
            $invoice = new Invoice();
            $invoice->invoice_id = $this->invoiceNumber();
            $invoice->customer_id = $request->customer_id;
            $invoice->status = 0;
            $invoice->issue_date = $request->issue_date;
            $invoice->due_date = $request->due_date;
            $invoice->category_id = $request->category_id;
            $invoice->ref_number = $request->ref_number;
            $invoice->created_by = \Auth::user()->creatorId();
            $invoice->save();
    
            // Loop through invoice items
            foreach ($request->items as $product) {
                if (empty($product['item'])) {
                    continue;
                }
    
                $productService = ProductService::find($product['item']);
                if (!$productService) {
                    continue;
                }
    
                // Create invoice product
                $invoiceProduct = new InvoiceProduct();
                $invoiceProduct->invoice_id = $invoice->id;
                $invoiceProduct->product_id = $product['item'];
                $invoiceProduct->quantity = floatval($product['quantity'] ?? 0);
                $invoiceProduct->discount = floatval($product['discount'] ?? 0);
                $invoiceProduct->price = floatval($product['price'] ?? 0);
                $invoiceProduct->description = $product['description'] ?? '';
                $invoiceProduct->tax = $productService->tax_id ? implode(',', (array) $productService->tax_id) : '';
                $invoiceProduct->save();
    
                // Update product stock
                Utility::total_quantity('minus', $invoiceProduct->quantity, $invoiceProduct->product_id);
    
                // Log stock change
                $description = "{$invoiceProduct->quantity} quantity sold in invoice " . \Auth::user()->invoiceNumberFormat($invoice->invoice_id);
                Utility::addProductStock($product['item'], $invoiceProduct->quantity, 'invoice', $description, $invoice->id);
            }
    
            return redirect()->route('invoice.index')->with('success', __('Invoice successfully created.'));
        } catch (\Exception $e) {

            return redirect()->back()->with('error', __('An error occurred. Please try again.'));
        }
    }
    public function edit($ids)
    {
        if (!\Auth::user()->can('edit invoice')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    
        try {
            $id = Crypt::decrypt($ids);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Something went wrong.'));
        }
    
        $invoice = Invoice::find($id);
        if (!$invoice) {
            return redirect()->back()->with('error', __('Invoice not found.'));
        }
    
        $invoice_number = \Auth::user()->invoiceNumberFormat($invoice->invoice_id);
    
        $customers = Customer::where('created_by', \Auth::user()->creatorId())
            ->pluck('name', 'id');
    
        $category = ProductServiceCategory::where('created_by', \Auth::user()->creatorId())
            ->where('type', 'income')
            ->pluck('name', 'id')
            ->prepend('Select Category', '');
    
        $product_services = ProductService::where('created_by', \Auth::user()->creatorId())
            ->pluck('name', 'id');
    
        $invoice->customField = CustomField::getData($invoice, 'invoice');
        $customFields = CustomField::where('created_by', \Auth::user()->creatorId())
            ->where('module', 'invoice')
            ->get();
    
        $settings_data = Utility::settingsById(\Auth::user()->creatorId());
        $taxType = $settings_data['tax_type'] ?? 'GST';
    
        $items = [];
        $processedIds = [];
    
        foreach ($invoice->items as $invoiceItem) {
            $invoiceItem->itemAmount = $invoiceItem->quantity * $invoiceItem->price;
            $invoiceItem->taxes = Utility::tax($invoiceItem->tax);
            $invoiceItem->pro_description = $invoiceItem->description ?? '';
    
            if ($taxType === 'VAT') {
                $vatRate = 0;
                foreach ($invoiceItem->taxes as $tax) {
                    $vatRate += $tax->rate ?? 0;
                }
                $invoiceItem->itemTaxRate = $vatRate;
            }else {
                // Default (e.g., GST) tax rate logic
                $invoiceItem->itemTaxRate = $invoiceItem->tax 
                    ? (json_decode($invoiceItem->tax, true)[0]['rate'] ?? 0) 
                    : 0;
            }
    
            if (!in_array($invoiceItem->product_id, $processedIds)) {
                $items[] = $invoiceItem;
                $processedIds[] = $invoiceItem->product_id;
            }
        }
    
        $customer = Customer::find($invoice->customer_id);
        $customerId = $invoice->customer_id;
    
        return view('invoice.edit', compact(
            'customers', 'product_services', 'invoice', 'invoice_number',
            'category', 'customFields', 'items', 'customer', 'customerId', 'settings_data'
        ));
    }
    

    public function invoiceNumber()
    {
        $latest = Invoice::where('created_by', '=', \Auth::user()->creatorId())->latest()->first();
        return $latest ? $latest->invoice_id + 1 : 1;
    }


    
    public function show($ids)
{
    if (!\Auth::user()->can('show invoice')) {
        return redirect()->back()->with('error', __('Permission denied.'));
    }

    try {
        $id = Crypt::decrypt($ids);
    } catch (\Throwable $th) {
       
        return redirect()->back()->with('error', __('Invoice Not Found.'));
    }

    $invoice = Invoice::with(['items.product.unit', 'payments', 'creditNote'])->find($id);

    if (!$invoice || $invoice->created_by != \Auth::user()->creatorId()) {
       
        return redirect()->back()->with('error', __('Permission denied.'));
    }

    $invoicePayment = $invoice->payments()->first();
    $customer = $invoice->customer;
    $items = $invoice->items;
    $user = \Auth::user();
    $invoice_user = User::find($invoice->created_by);
    $user_plan = Plan::getPlan($invoice_user->plan);

    $invoice->customField = CustomField::getData($invoice, 'invoice');
    $customFields = CustomField::where('created_by', \Auth::user()->creatorId())
        ->where('module', 'invoice')
        ->get();
    $creditnote = $invoice->creditNote()->first();

    // Determine state and tax logic
    $companyState = strtolower(trim(Utility::settingsById($invoice->created_by)['company_state'] ?? ''));
    $billingState = strtolower(trim($customer->billing_state ?? ''));
    $isSameState = $companyState && $billingState && $companyState === $billingState;

    // Tax Calculation
    $taxesData = ['CGST' => 0, 'SGST' => 0, 'IGST' => 0];
    $itemData = [];

    foreach ($invoice->items as $invItem) {
        $product = $invItem->product;
        $item = new \stdClass();
        $item->name = $product->name ?? '';
        $item->quantity = $invItem->quantity;
        $item->price = $invItem->price;
        $item->discount = $invItem->discount;
        $item->unit = $product->unit_id ?? '';
        $item->description = $invItem->description;
        $item->itemTax = [];

        $baseAmount = ($item->price - $item->discount) * $item->quantity;

        $taxRates = Utility::tax($invItem->tax);

        foreach ($taxRates as $tax) {
    if (isset($tax->rate)) {
        $rate = floatval($tax->rate);
        $taxAmount = Utility::taxRate($rate, $item->price, $item->quantity, $item->discount);

        // Check for VAT tax type
        if (isset($tax->type) && strtolower($tax->type) == 'vat') {
            $item->itemTax[] = [
                'name' => 'VAT',
                'rate' => $rate . '%',
                'tax_price' => $taxAmount,
            ];
            $taxesData['VAT'] = ($taxesData['VAT'] ?? 0) + $taxAmount;
        } else {
            // Default to GST
            if ($isSameState) {
                $half = $taxAmount / 2;

                $item->itemTax[] = ['name' => 'CGST', 'rate' => $rate . '%', 'tax_price' => $half];
                $item->itemTax[] = ['name' => 'SGST', 'rate' => $rate . '%', 'tax_price' => $half];

                $taxesData['CGST'] += $half;
                $taxesData['SGST'] += $half;
            } else {
                $item->itemTax[] = ['name' => 'IGST', 'rate' => $rate . '%', 'tax_price' => $taxAmount];
                $taxesData['IGST'] += $taxAmount;
            }
        }
    }
}


        $itemData[] = $item;
    }

    $invoice->itemData = $itemData;

    return view('invoice.view', compact(
        'invoice', 'customer', 'items', 'invoicePayment',
        'customFields', 'user', 'invoice_user', 'user_plan',
        'creditnote', 'taxesData', 'isSameState'
    ));
}

    public function destroy(Invoice $invoice, Request $request)
    {
        if (\Auth::user()->can('delete invoice')) {
            if ($invoice->created_by == \Auth::user()->creatorId()) {
                foreach ($invoice->payments as $payment) {
                    Utility::bankAccountBalance($payment->account_id, $payment->amount, 'debit');
                    $payment->delete();
                }

                if ($invoice->customer_id != 0 && $invoice->status != 0) {
                    Utility::updateUserBalance('customer', $invoice->customer_id, $invoice->getDue(), 'debit');
                }

                TransactionLines::where('reference_id', $invoice->id)->where('reference', 'Invoice')->delete();
                TransactionLines::where('reference_id', $invoice->id)->where('reference', 'Invoice Payment')->delete();
                CreditNote::where('invoice', '=', $invoice->id)->delete();
                InvoiceProduct::where('invoice_id', '=', $invoice->id)->delete();
                $invoice->delete();

                return redirect()->route('invoice.index')->with('success', __('Invoice successfully deleted.'));
            } else {
                return redirect()->back()->with('error', __('Permission denied.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function productDestroy(Request $request)
    {
        if (\Auth::user()->can('delete invoice product')) {
            $invoiceProduct = InvoiceProduct::find($request->id);

            if ($invoiceProduct) {
                $invoice = Invoice::find($invoiceProduct->invoice_id);
                $productService = ProductService::find($invoiceProduct->product_id);

                Utility::updateUserBalance('customer', $invoice->customer_id, $request->amount, 'debit');
                TransactionLines::where('reference_sub_id', $productService->id)->where('reference', 'Invoice')->delete();
                $invoiceProduct->delete();

                return response()->json(['status' => true, 'message' => __('Invoice product successfully deleted.')]);
            } else {
                return response()->json(['status' => false, 'message' => __('Invoice product not found!')]);
            }
        } else {
            return response()->json(['status' => false, 'message' => __('Permission denied.')]);
        }
    }

    public function customerInvoice(Request $request)
    {
        if (\Auth::user()->can('manage customer invoice')) {
            $status = Invoice::$statues;
            $query = Invoice::where('customer_id', '=', \Auth::user()->id)->where('status', '!=', '0')->where('created_by', \Auth::user()->creatorId());

            if (!empty($request->issue_date)) {
                $date_range = explode(' - ', $request->issue_date);
                $query->whereBetween('issue_date', $date_range);
            }

            if (!empty($request->status)) {
                $query->where('status', '=', $request->status);
            }
            $invoices = $query->get();

            return view('invoice.index', compact('invoices', 'status'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }
    public function customerInvoiceShow($id)
    {
        if (\Auth::user()->can('show invoice')) {
            try {
                $invoice = Invoice::with('payments.bankAccount', 'items.product.unit')->find($id);
    
                if (!$invoice) {
                   
                    return redirect()->back()->with('error', __('Invoice not found.'));
                }
    
                $user = User::find($invoice->created_by);
    
                if ($invoice->created_by != $user->creatorId()) {
                   
                    return redirect()->back()->with('error', __('Permission denied.'));
                }
    
                $customer = $invoice->customer;
                $iteams = $invoice->items ?? collect([]);
    
                if (!$customer) {
                    \Log::warning('Customer not found', ['customer_id' => $invoice->customer_id]);
                }
    
                $rawItems = \DB::table('invoice_products')
                    ->where('invoice_id', $id)
                    ->get()
                    ->toArray();
    
                $settings = DB::table('settings')
                    ->where('created_by', $user->creatorId())
                    ->pluck('value', 'name')
                    ->toArray();
    
                $companySettings = Utility::settingsById($invoice->created_by);
                $customFields = \App\Models\CustomField::where('created_by', $invoice->created_by)->get();
    
                $companyState = strtolower(trim($companySettings['company_state'] ?? ''));
                $billingState = strtolower(trim($customer->billing_state ?? ''));
                $isSameState = $companyState && $billingState && $companyState === $billingState;
    
                $view = $user->type === 'company' ? 'invoice.customer_invoice' : 'invoice.view';
                return view($view, compact(
                    'invoice', 'customer', 'iteams', 'user', 'settings',
                    'companySettings', 'customFields', 'isSameState'
                ));
            } catch (\Exception $e) {
               
                return redirect()->back()->with('error', __('An error occurred while processing the invoice. Please try again.'));
            }
        } else {
           
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
    
    public function sent($id)
    {
        if (\Auth::user()->can('send invoice')) {
            $setings = Utility::settings();

            if ($setings['customer_invoice_sent'] == 1) {
                $invoice = Invoice::where('id', $id)->first();
                $invoice->send_date = date('Y-m-d');
                $invoice->status = 1;
                $invoice->save();

                $customer = Customer::where('id', $invoice->customer_id)->first();
                $invoice->name = !empty($customer) ? $customer->name : '';
                $invoice->invoice = \Auth::user()->invoiceNumberFormat($invoice->invoice_id);
                $invoiceId = Crypt::encrypt($invoice->id);
                $invoice->url = route('invoice.pdf', $invoiceId);

                $invoice_products = InvoiceProduct::where('invoice_id', $invoice->id)->get();
                foreach ($invoice_products as $invoice_product) {
                    $product = ProductService::find($invoice_product->product_id);
                    $totalTaxPrice = 0;

                    if ($invoice_product->tax != null) {
                        $taxPrice = Utility::taxRate($invoice_product->tax, $invoice_product->price, $invoice_product->quantity, $invoice_product->discount);
                        $totalTaxPrice += $taxPrice;
                    }

                    $itemAmount = ($invoice_product->price * $invoice_product->quantity) - ($invoice_product->discount) + $totalTaxPrice;

                    $data = [
                        'account_id' => $product->sale_chartaccount_id,
                        'transaction_type' => 'Credit',
                        'transaction_amount' => $itemAmount,
                        'reference' => 'Invoice',
                        'reference_id' => $invoice->id,
                        'reference_sub_id' => $product->id,
                        'date' => $invoice->issue_date,
                    ];
                    Utility::addTransactionLines($data, 'create');
                }

                $customerArr = [
                    'customer_name' => $customer->name,
                    'customer_email' => $customer->email,
                    'invoice_name' => $customer->name,
                    'invoice_number' => $invoice->invoice,
                    'invoice_url' => $invoice->url,
                ];
                $resp = Utility::sendEmailTemplate('customer_invoice_sent', [$customer->id => $customer->email], $customerArr);

                if (!empty($customer->contact)) {
                    $number = preg_replace('/[^0-9]/', '', $customer->contact);
                    $message = "Dear {$customer->name}, your invoice ({$invoice->invoice}) is ready. Please check your email.";
                    $whatsappStatus = Utility::sendWhatsappMessage($number, $message);
            
                } 

                return redirect()->back()->with('success', __('Invoice successfully sent.') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
            } else {
                return redirect()->back()->with('error', 'Invoice not found');
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function resent($id)
    {
        if (\Auth::user()->can('send invoice')) {
            $invoice = Invoice::where('id', $id)->first();
            $customer = Customer::where('id', $invoice->customer_id)->first();
            $invoice->name = !empty($customer) ? $customer->name : '';
            $invoice->invoice = \Auth::user()->invoiceNumberFormat($invoice->invoice_id);
            $invoiceId = Crypt::encrypt($invoice->id);
            $invoice->url = route('invoice.pdf', $invoiceId);

            $customerArr = [
                'customer_name' => $customer->name,
                'customer_email' => $customer->email,
                'invoice_name' => $customer->name,
                'invoice_number' => $invoice->invoice,
                'invoice_url' => $invoice->url,
            ];
            $resp = Utility::sendEmailTemplate('customer_invoice_sent', [$customer->id => $customer->email], $customerArr);

            if (!empty($customer->contact)) {
                $number = preg_replace('/[^0-9]/', '', $customer->contact);
                $message = "Dear {$customer->name}, your invoice ({$invoice->invoice}) is ready. Please check your email.";
                $whatsappStatus = Utility::sendWhatsappMessage($number, $message);
        
            } 

            return redirect()->back()->with('success', __('Invoice successfully sent.') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function payment($invoice_id)
    {
        if (\Auth::user()->can('create payment invoice')) {
            $invoice = Invoice::where('id', $invoice_id)->first();
            $customers = Customer::where('created_by', '=', \Auth::user()->creatorId())->get()->pluck('name', 'id');
            $categories = ProductServiceCategory::where('created_by', '=', \Auth::user()->creatorId())->get()->pluck('name', 'id');
            $accounts = BankAccount::select('*', \DB::raw("CONCAT(bank_name,' ',holder_name) AS name"))->where('created_by', \Auth::user()->creatorId())->get()->pluck('name', 'id');

            return view('invoice.payment', compact('customers', 'categories', 'accounts', 'invoice'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function createPayment(Request $request, $invoice_id)
    {
        $invoice = Invoice::find($invoice_id);
        if ($invoice->getDue() < $request->amount) {
            return redirect()->back()->with('error', __('Invoice payment amount should not greater than subtotal.'));
        }

        if (\Auth::user()->can('create payment invoice')) {
            $validator = \Validator::make(
                $request->all(),
                [
                    'date' => 'required',
                    'amount' => 'required|numeric|min:0',
                    'account_id' => 'required',
                ]
            );
            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $invoicePayment = new InvoicePayment();
            $invoicePayment->invoice_id = $invoice_id;
            $invoicePayment->date = $request->date;
            $invoicePayment->amount = floatval($request->amount);
            $invoicePayment->account_id = $request->account_id;
            $invoicePayment->payment_method = 0;
            $invoicePayment->reference = $request->reference;
            $invoicePayment->description = $request->description;

            if (!empty($request->add_receipt)) {
                $image_size = $request->file('add_receipt')->getSize();
                $result = Utility::updateStorageLimit(\Auth::user()->creatorId(), $image_size);
                if ($result == 1) {
                    $fileName = time() . "_" . $request->add_receipt->getClientOriginalName();
                    $request->add_receipt->storeAs('uploads/payment', $fileName);
                    $invoicePayment->add_receipt = $fileName;
                }
            }

            $invoicePayment->save();

            $invoice = Invoice::where('id', $invoice_id)->first();
            $due = $invoice->getDue();
            $total = $invoice->getTotal();
            if ($invoice->status == 0) {
                $invoice->send_date = date('Y-m-d');
                $invoice->save();
            }

            if ($due <= 0) {
                $invoice->status = 4;
                $invoice->save();
            } else {
                $invoice->status = 3;
                $invoice->save();
            }

            $invoicePayment->user_id = $invoice->customer_id;
            $invoicePayment->user_type = 'Customer';
            $invoicePayment->type = 'Partial';
            $invoicePayment->created_by = \Auth::user()->id;
            $invoicePayment->payment_id = $invoicePayment->id;
            $invoicePayment->category = 'Invoice';
            $invoicePayment->account = $request->account_id;

            Transaction::addTransaction($invoicePayment);
            $customer = Customer::where('id', $invoice->customer_id)->first();

            $payment = new InvoicePayment();
            $payment->name = $customer['name'];
            $payment->date = \Auth::user()->dateFormat($request->date);
            $payment->amount = \Auth::user()->priceFormat($request->amount);
            $payment->invoice = 'invoice ' . \Auth::user()->invoiceNumberFormat($invoice->invoice_id);
            $payment->dueAmount = \Auth::user()->priceFormat($invoice->getDue());

            Utility::updateUserBalance('customer', $invoice->customer_id, $request->amount, 'credit');
            Utility::bankAccountBalance($request->account_id, $request->amount, 'credit');

            $invoicePayments = InvoicePayment::where('invoice_id', $invoice->id)->get();
            foreach ($invoicePayments as $invoicePayment) {
                $accountId = BankAccount::find($invoicePayment->account_id);
                $data = [
                    'account_id' => $accountId->chart_account_id,
                    'transaction_type' => 'Debit',
                    'transaction_amount' => $invoicePayment->amount,
                    'reference' => 'Invoice Payment',
                    'reference_id' => $invoice->id,
                    'reference_sub_id' => $invoicePayment->id,
                    'date' => $invoicePayment->date,
                ];
                Utility::addTransactionLines($data, 'create');
            }

            $setings = Utility::settings();
            if ($setings['new_invoice_payment'] == 1) {
                $customer = Customer::where('id', $invoice->customer_id)->first();
                $invoice_payment_method = InvoicePayment::where('invoice_id', $invoice->id)->orderBy('id', 'desc')->first();
                $invoicePaymentArr = [
                    'invoice_payment_name' => $customer->name,
                    'invoice_payment_amount' => $payment->amount,
                    'invoice_payment_date' => $payment->date,
                    'payment_dueAmount' => $payment->dueAmount,
                    'invoice_number' => \Auth::user()->priceFormat($invoice->getDue()),
                    'invoice_payment_method' => $invoice_payment_method->payment_type,
                ];
                $resp = Utility::sendEmailTemplate('new_invoice_payment', [$customer->id => $customer->email], $invoicePaymentArr);

                if (!empty($customer->contact)) {
                    $number = preg_replace('/[^0-9]/', '', $customer->contact);
                
                    $amount = floatval($request->amount);
                    $due = $invoice->getDue(); // This returns numeric value
                
                    $message = "Dear {$customer->name},\n"
                        . "Thank you for your payment of " . \Auth::user()->priceFormat($amount) . " on " . date('d-m-Y', strtotime($request->date)) . ".\n"
                        . "Invoice Number: {$invoice->invoice_id}\n"
                        . "Payment Method: {$invoice_payment_method->payment_type}\n"
                        . "Remaining Due: " . \Auth::user()->priceFormat($due);
                
                    $whatsappStatus = Utility::sendWhatsappMessage($number, $message);
                }
                
            }

            $module = 'New Invoice Payment';
            $webhook = Utility::webhookSetting($module);
            if ($webhook) {
                $parameter = json_encode($invoice);
                $status = Utility::WebhookCall($webhook['url'], $parameter, $webhook['method']);
                if ($status == true) {
                    return redirect()->back()->with('success', __('Payment successfully added.') . ((isset($result) && $result != 1) ? '<br> <span class="text-danger">' . $result . '</span>' : '') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
                } else {
                    return redirect()->back()->with('error', __('Payment successfully, Webhook call failed.'));
                }
            }
            return redirect()->back()->with('success', __('Payment successfully added.') . ((isset($result) && $result != 1) ? '<br> <span class="text-danger">' . $result . '</span>' : '') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
        }
    }

    public function paymentDestroy(Request $request, $invoice_id, $payment_id)
    {
        if (\Auth::user()->can('delete payment invoice')) {
            $payment = InvoicePayment::find($payment_id);
            InvoicePayment::where('id', '=', $payment_id)->delete();
            InvoiceBankTransfer::where('id', '=', $payment_id)->delete();
            TransactionLines::where('reference_sub_id', $payment_id)->where('reference', 'Invoice Payment')->delete();

            $invoice = Invoice::where('id', $invoice_id)->first();
            $due = $invoice->getDue();
            $total = $invoice->getTotal();

            if ($due > 0 && $total != $due) {
                $invoice->status = 3;
            } else {
                $invoice->status = 2;
            }

            if (!empty($payment->add_receipt)) {
                $file_path = '/uploads/payment/' . $payment->add_receipt;
                Utility::changeStorageLimit(\Auth::user()->creatorId(), $file_path);
            }

            $invoice->save();
            $type = 'Partial';
            $user = 'Customer';
            Transaction::destroyTransaction($payment_id, $type, $user);

          Utility::updateUserBalance('customer', $invoice->customer_id, $payment->amount, 'debit');
            Utility::bankAccountBalance($payment->account_id, $payment->amount, 'debit');

            return redirect()->back()->with('success', __('Payment successfully deleted.'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function paymentReminder($invoice_id)
    {
        $invoice = Invoice::find($invoice_id);
        $customer = Customer::where('id', $invoice->customer_id)->first();
        $invoice->dueAmount = \Auth::user()->priceFormat($invoice->getDue());
        $invoice->name = $customer['name'];
        $invoice->date = \Auth::user()->dateFormat($invoice->send_date);
        $invoice->invoice = \Auth::user()->invoiceNumberFormat($invoice->invoice_id);

        $setting = Utility::settings(\Auth::user()->creatorId());
        $reminderNotificationArr = [
            'invoice_number' => \Auth::user()->invoiceNumberFormat($invoice->invoice_id),
            'customer_name' => $customer->name,
            'user_name' => \Auth::user()->name,
        ];

        if (isset($setting['twilio_reminder_notification']) && $setting['twilio_reminder_notification'] == 1) {
            Utility::send_twilio_msg($customer->contact, 'invoice_payment_reminder', $reminderNotificationArr);
        }

        $setings = Utility::settings();
        if ($setings['new_payment_reminder'] == 1) {
            $reminderArr = [
                'payment_reminder_name' => $invoice->name,
                'invoice_payment_number' => $invoice->invoice,
                'invoice_payment_dueAmount' => $invoice->dueAmount,
                'payment_reminder_date' => $invoice->date,
            ];
            $resp = Utility::sendEmailTemplate('new_payment_reminder', [$customer->id => $customer->email], $reminderArr);

            if (!empty($customer->contact)) {
                $number = preg_replace('/[^0-9]/', '', $customer->contact);
            
                $dueAmount = \Auth::user()->priceFormat($invoice->getDue());
                $dueDate = date('d-m-Y', strtotime($invoice->due_date ?? $invoice->date)); // use `due_date` if available
            
                $message = "Dear {$customer->name},\n"
                    . "This is a gentle reminder that payment for Invoice #{$invoice->invoice_id} is due.\n"
                    . "Due Amount: {$dueAmount}\n"
                    . "Due Date: {$dueDate}";
            
                $whatsappStatus = Utility::sendWhatsappMessage($number, $message);
            }
        }

        return redirect()->back()->with('success', __('Payment reminder successfully send.') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
    }

    public function customerInvoiceSend($invoice_id)
    {
        return view('customer.invoice_send', compact('invoice_id'));
    }

    public function customerInvoiceSendMail(Request $request, $invoice_id)
    {
        $validator = \Validator::make(
            $request->all(),
            ['email' => 'required|email']
        );
        if ($validator->fails()) {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error', $messages->first());
        }

        $email = $request->email;
        $invoice = Invoice::where('id', $invoice_id)->first();
        $customer = Customer::where('id', $invoice->customer_id)->first();
        $invoice->name = !empty($customer) ? $customer->name : '';
        $invoice->invoice = \Auth::user()->invoiceNumberFormat($invoice->invoice_id);
        $invoiceId = Crypt::encrypt($invoice->id);
        $invoice->url = route('invoice.pdf', $invoiceId);

        try {
            Mail::to($email)->send(new CustomerInvoiceSend($invoice));
        } catch (\Exception $e) {
            $smtp_error = __('E-Mail has been not sent due to SMTP configuration');
        }

        return redirect()->back()->with('success', __('Invoice successfully sent.') . ((isset($smtp_error)) ? '<br> <span class="text-danger">' . $smtp_error . '</span>' : ''));
    }

    public function shippingDisplay(Request $request, $id)
    {
        $invoice = Invoice::find($id);
        $invoice->shipping_display = $request->is_display == 'true' ? 1 : 0;
        $invoice->save();

        return redirect()->back()->with('success', __('Shipping address status successfully changed.'));
    }

    public function duplicate($invoice_id)
    {
        if (\Auth::user()->can('duplicate invoice')) {
            $invoice = Invoice::where('id', $invoice_id)->first();
            $duplicateInvoice = new Invoice();
            $duplicateInvoice->invoice_id = $this->invoiceNumber();
            $duplicateInvoice->customer_id = $invoice->customer_id;
            $duplicateInvoice->issue_date = date('Y-m-d');
            $duplicateInvoice->due_date = $invoice->due_date;
            $duplicateInvoice->send_date = null;
            $duplicateInvoice->category_id = $invoice->category_id;
            $duplicateInvoice->ref_number = $invoice->ref_number;
            $duplicateInvoice->status = 0;
            $duplicateInvoice->shipping_display = $invoice->shipping_display;
            $duplicateInvoice->created_by = $invoice->created_by;
            $duplicateInvoice->save();

            $invoiceProduct = InvoiceProduct::where('invoice_id', $invoice_id)->get();
            foreach ($invoiceProduct as $product) {
                $duplicateProduct = new InvoiceProduct();
                $duplicateProduct->invoice_id = $duplicateInvoice->id;
                $duplicateProduct->product_id = $product->product_id;
                $duplicateProduct->quantity = $product->quantity;
                $duplicateProduct->tax = $product->tax;
                $duplicateProduct->discount = $product->discount;
                $duplicateProduct->price = $product->price;
                $duplicateProduct->description = $product->description;
                $duplicateProduct->save();
            }

            return redirect()->back()->with('success', __('Invoice duplicate successfully.'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function previewInvoice($template, $color)
    {
        $objUser = \Auth::user();
        $settings = Utility::settings();
        $invoice = new Invoice();

        $customer = new \stdClass();
        $customer->email = '<Email>';
        $customer->shipping_name = '<Customer Name>';
        $customer->shipping_country = '<Country>';
        $customer->shipping_state = '<State>';
        $customer->shipping_city = '<City>';
        $customer->shipping_phone = '<Customer Phone Number>';
        $customer->shipping_zip = '<Zip>';
        $customer->shipping_address = '<Address>';
        $customer->billing_name = '<Customer Name>';
        $customer->billing_country = '<Country>';
        $customer->billing_state = '<State>';
        $customer->billing_city = '<City>';
        $customer->billing_phone = '<Customer Phone Number>';
        $customer->billing_zip = '<Zip>';
        $customer->billing_address = '<Address>';

        $totalTaxPrice = 0;
        $taxesData = [];
        $items = [];
        for ($i = 1; $i <= 3; $i++) {
            $item = new \stdClass();
            $item->name = 'Item ' . $i;
            $item->quantity = 1;
            $item->tax = 5;
            $item->discount = 50;
            $item->price = 100;
            $item->unit = 1;
            $item->description = 'XYZ';

            $taxes = [
                'Tax 1',
                'Tax 2',
            ];

            $itemTaxes = [];
            foreach ($taxes as $k => $tax) {
                $taxPrice = 10;
                $totalTaxPrice += $taxPrice;
                $itemTax['name'] = 'Tax ' . $k;
                $itemTax['rate'] = '10 %';
                $itemTax['price'] = '$10';
                $itemTax['tax_price'] = 10;
                $itemTaxes[] = $itemTax;
                $taxesData['Tax ' . $k] = ($taxesData['Tax ' . $k] ?? 0) + $taxPrice;
            }
            $item->itemTax = $itemTaxes;
            $items[] = $item;
        }

        $invoice->invoice_id = 1;
        $invoice->issue_date = date('Y-m-d H:i:s');
        $invoice->due_date = date('Y-m-d H:i:s');
        $invoice->itemData = $items;
        $invoice->status = 0;
        $invoice->totalTaxPrice = 60;
        $invoice->totalQuantity = 3;
        $invoice->totalRate = 300;
        $invoice->totalDiscount = 10;
        $invoice->taxesData = $taxesData;
        $invoice->created_by = $objUser->creatorId();
        $accounts = BankAccount::where('created_by', $objUser->creatorId())->get();

       $invoicePayment = \App\Models\InvoicePayment::with('bankAccount')
    ->where('invoice_id', $invoice->id)
    ->latest()
    ->first(); 


        $invoice->customField = [];
        $customFields = [];

        $preview = 1;
        $color = '#' . $color;
      $fontColor = Utility::getFontColor($color);

    
        $logo = asset(Storage::url('uploads/logo/'));
        $company_logo = Utility::getValByName('company_logo_dark');
        
        $invoice_logo = Utility::getValByName('invoice_logo');
     $img = isset($invoice_logo) && !empty($invoice_logo)
    ? asset('storage/invoice_logo/' . $invoice_logo)
    : asset($logo . '/' . (isset($company_logo) && !empty($company_logo) ? $company_logo : 'logo-dark.png'));




        return view('invoice.templates.' . $template, compact('invoice', 'preview', 'color', 'img', 'settings', 'customer', 'fontColor', 'customFields', 'accounts','invoicePayment',));
    }

 public function invoice($invoice_id)
{
    try {
        $settings = Utility::settings();
        $invoiceId = Crypt::decrypt($invoice_id);
        $invoice = Invoice::with('items.product')->find($invoiceId);

        if (!$invoice) {
            Log::warning('Invoice not found', ['invoice_id' => $invoiceId]);
            return redirect()->back()->with('error', __('Invoice not found.'));
        }

        $creatorSettings = DB::table('settings')
            ->where('created_by', $invoice->created_by)
            ->pluck('value', 'name')
            ->toArray();
        $settings = array_merge($settings, $creatorSettings);

        $customer = $invoice->customer;
        if (!$customer) {
            Log::warning('Customer not found', ['customer_id' => $invoice->customer_id]);
        }

        $companyState = strtolower(trim(Utility::settingsById($invoice->created_by)['company_state'] ?? ''));
        $billingState = strtolower(trim($customer->billing_state ?? ''));
        $isSameState = ($companyState && $billingState && $companyState === $billingState);

        $items = [];
        $taxesData = [
            'CGST' => 0,
            'SGST' => 0,
            'IGST' => 0,
            'VAT'  => 0,
        ];
        $totalTax = 0;
        $totalQuantity = 0;
        $totalRate = 0;
        $totalDiscount = 0;

        foreach ($invoice->items as $invItem) {
            $item = new \stdClass();
            $product = $invItem->product;

            $item->name = $product->name ?? '';
            $item->quantity = $invItem->quantity ?? 0;
            $item->unit = $product->unit_id ?? '';
            $item->price = $invItem->price ?? 0;
            $item->discount = $invItem->discount ?? 0;
            $item->description = $invItem->description ?? '';
            $item->expense_chartaccount_code = $product->expenseChartAccount->code ?? null;

            $totalQuantity += $item->quantity;
            $totalRate += $item->quantity * $item->price;
            $totalDiscount += $item->discount;

            $itemTaxes = [];
            $itemVATTotal = 0;

            if ($product && $invItem->tax) {
                $taxRates = Utility::tax($invItem->tax);
                if (!empty($taxRates)) {
                    foreach ($taxRates as $tax) {
                        if ($tax && isset($tax->rate)) {
                            $rate = floatval($tax->rate);
                            $taxAmount = Utility::taxRate($rate, $item->price, $item->quantity, $item->discount);

                         if (!empty($settings['tax_type']) && $settings['tax_type'] == 'VAT') {
    $taxesData['VAT'] += $taxAmount;
    $totalTax += $taxAmount;

    $itemTaxes[] = [
        'name' => 'VAT',
        'rate' => ($rate * 2) . '%',
        'price' => Utility::priceFormat($settings, $taxAmount),
        'tax_price' => $taxAmount,
    ];


                            } elseif ($isSameState) {
                                $half = $taxAmount / 2;
                                $itemTaxes[] = [
                                    'name' => 'CGST',
                                    'rate' => $rate . '%',
                                    'price' => Utility::priceFormat($settings, $half),
                                    'tax_price' => $half,
                                ];
                                $itemTaxes[] = [
                                    'name' => 'SGST',
                                    'rate' => $rate . '%',
                                    'price' => Utility::priceFormat($settings, $half),
                                    'tax_price' => $half,
                                ];
                                $taxesData['CGST'] += $half;
                                $taxesData['SGST'] += $half;
                            } else {
                                $itemTaxes[] = [
                                    'name' => 'IGST',
                                    'rate' => ($rate * 2) . '%',
                                    'price' => Utility::priceFormat($settings, $taxAmount),
                                    'tax_price' => $taxAmount,
                                ];
                                $taxesData['IGST'] += $taxAmount;
                            }
                        } else {
                            Log::warning('Invalid tax object', ['tax' => $tax]);
                        }
                    }

                    // Apply VAT per item only once
                    if ($itemVATTotal > 0) {
                        $taxesData['VAT'] += $itemVATTotal;


                    }
                }
            }

            $item->itemTax = $itemTaxes;
            $items[] = $item;
        }

        $invoice->itemData = $items;
        $invoice->taxesData = array_filter($taxesData, fn($value) => $value > 0);
        $invoice->totalTaxPrice = $totalTax;
        $invoice->totalRate = $totalRate;
        $invoice->totalDiscount = $totalDiscount;
        $invoice->customField = CustomField::getData($invoice, 'invoice');

        $customFields = auth()->check()
            ? CustomField::where('created_by', auth()->user()->creatorId())->where('module', 'invoice')->get()
            : [];

        $accounts = BankAccount::where('created_by', $invoice->created_by)->get();
         
        $invoicePayment = \App\Models\InvoicePayment::with('bankAccount')
    ->where('invoice_id', $invoice->id)
    ->latest()
    ->first(); 

        $logoPath = asset(Storage::url('uploads/logo/'));
        $invoiceLogo = Utility::settingsById($invoice->created_by)['invoice_logo'] ?? '';
        $companyLogo = Utility::getValByName('company_logo_dark');
        $img = $invoiceLogo
            ? Utility::get_file('invoice_logo/') . $invoiceLogo
            : $logoPath . '/' . ($companyLogo ?: 'logo-dark.png');

        $color = '#' . ($settings['invoice_color'] ?? '000000');
        $fontColor = Utility::getFontColor($color);

        return view('invoice.templates.' . ($settings['invoice_template'] ?? 'default'), compact(
            'invoice', 'settings', 'customer', 'img', 'color', 'fontColor', 'customFields', 'accounts','taxesData','invoicePayment',
        ));
    } catch (\Exception $e) {
       
        return redirect()->back()->with('error', __('An error occurred while generating invoice.'));
    }
}


public function saveTemplateSettings(Request $request)
{
    $post = $request->all();
    unset($post['_token']);

    if (isset($post['invoice_template']) && (!isset($post['invoice_color']) || empty($post['invoice_color']))) {
        $post['invoice_color'] = "ffffff";
    }

    if ($request->hasFile('invoice_logo')) {
        $dir = 'invoice_logo/';
        $invoice_logo = \Auth::user()->id . '_invoice_logo.png';
        $validation = [
            'mimes:png', // Correct format for Laravel validation
            'max:20480', // Max size 20MB
        ];
        $path = Utility::upload_file($request, 'invoice_logo', $invoice_logo, $dir, $validation);

        if ($path['flag'] == 0) {
            return redirect()->back()->with('error', __($path['msg']));
        }
        $post['invoice_logo'] = $invoice_logo;
    }

    foreach ($post as $key => $data) {
        \DB::insert(
            'insert into settings (`value`, `name`, `created_by`) values (?, ?, ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)',
            [$data, $key, \Auth::user()->creatorId()]
        );
    }

    return redirect()->back()->with('success', __('Invoice Setting updated successfully'));
}
    public function items(Request $request)
    {
        $items = InvoiceProduct::where('invoice_id', $request->invoice_id)->where('product_id', $request->product_id)->first();
        return json_encode($items);
    }

    public function invoiceLink($invoiceId)
    {
        try {
            $id = Crypt::decrypt($invoiceId);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', __('Invoice Not Found.'));
        }

        $invoice = Invoice::with(['creditNote', 'payments.bankAccount', 'items.product.unit'])->find($id);
        $settings = Utility::settingsById($invoice->created_by);

        if (!empty($invoice)) {
            $user_id = $invoice->created_by;
            $user = User::find($user_id);
            $invoicePayment = InvoicePayment::where('invoice_id', $invoice->id)->get();
            $customer = $invoice->customer;
            $iteams = $invoice->items;
            $invoice->customField = CustomField::getData($invoice, 'invoice');
            $customFields = CustomField::where('module', '=', 'invoice')->where('created_by', $invoice->created_by)->get();
            $company_payment_setting = Utility::getCompanyPaymentSetting($user_id);
            $user_plan = Plan::find($user->plan);

            return view('invoice.customer_invoice', compact('settings', 'invoice', 'customer', 'iteams', 'invoicePayment', 'customFields', 'user', 'company_payment_setting', 'user_plan'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }

    public function export()
    {
        $name = 'invoice_' . date('Y-m-d i:h:s');
        $data = Excel::download(new InvoiceExport(), $name . '.xlsx');
        ob_end_clean();
        return $data;
    }

      public function convert($invoice_id)
{
    if (\Auth::user()->can('convert invoice')) {
        $invoice = Invoice::where('id', $invoice_id)->first();
        if (!$invoice) {
            return redirect()->back()->with('error', __('Invoice not found.'));
        }

        $invoice->is_convert = 1;
        $invoice->save();

        $convertewaybill = new EWayBill();
        $convertewaybill->ewaybill_id = $this->eBillNumber();
        $convertewaybill->customer_id = $invoice->customer_id;
        $convertewaybill->issue_date = date('Y-m-d');
        $convertewaybill->due_date = date('Y-m-d');
        $convertewaybill->send_date = null;
        $convertewaybill->category_id = $invoice->category_id;
        $convertewaybill->status = 0;
        $convertewaybill->created_by = $invoice->created_by;
        $convertewaybill->save();

        $invoice->converted_ewayBill_id = $convertewaybill->id;
        $invoice->save();

        if ($convertewaybill) {
            $invoiceProduct = InvoiceProduct::where('invoice_id', $invoice_id)->get();
            foreach ($invoiceProduct as $product) {
                $duplicateProduct = new EWayBillProduct();
                $duplicateProduct->ewaybill_id = $convertewaybill->id;
                $duplicateProduct->product_id = $product->product_id;
                $duplicateProduct->quantity = $product->quantity;
                $duplicateProduct->tax = $product->tax;
                $duplicateProduct->discount = $product->discount;
                $duplicateProduct->price = $product->price;
                $duplicateProduct->save();

                Utility::total_quantity('minus', $duplicateProduct->quantity, $duplicateProduct->product_id);

                $type = 'ewaybill';
                $type_id = $convertewaybill->id;
                StockReport::where('type', '=', 'ewaybill')->where('type_id', '=', $convertewaybill->id)->delete();
                // Retrieve settings
               $description = $duplicateProduct->quantity . __(' quantity sold in') . ' ' . \Auth::user()->invoiceNumberFormat($invoice->invoice_id) . ' ' . __('Invoice convert to ewaybill') . ' ' . \Auth::user()->eBillNumberFormat($convertewaybill->ewaybill_id);
                Utility::addProductStock($duplicateProduct->product_id, $duplicateProduct->quantity, $type, $description, $type_id);
            }
        }

        return redirect()->back()->with('success', __('Invoice to ewaybill convert successfully.'));
    } else {
        return redirect()->back()->with('error', __('Permission denied.'));
    }
}

function eBillNumber()
{
   $latest = EWayBill::where('created_by', '=', \Auth::user()->creatorId())->latest()->first();
        return $latest ? $latest->ewaybill_id + 1 : 1;
}
}