<?php

namespace App\Http\Controllers;

use App\Exports\EWayBillExport;
use App\Models\BankAccount;
use App\Models\CreditNote;
use App\Models\Customer;
use App\Models\CustomField;
use App\Models\EWayBill;
use App\Models\InvoiceBankTransfer;
use App\Models\InvoicePayment;
use App\Models\EWayBillProduct;
use App\Models\Plan;
use App\Models\ProductService;
use App\Models\ProductServiceCategory;
use App\Models\StockReport;
use App\Models\Transaction;
use App\Models\Utility;
use App\Models\TransactionLines;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;

class EWayBillController extends Controller
{
    public function __construct()
    {
    }

    public function index(Request $request)
    {
        if (\Auth::user()->can('manage invoice')) {
        $customer = Customer::where('created_by', '=', \Auth::user()->creatorId())->get()->pluck('name', 'id');
        $customer->prepend('Select Customer', '');
        $status = EWayBill::$statues;
        $query =EWayBill::where('created_by', '=', \Auth::user()->creatorId());

        if (!empty($request->customer)) {
            $query->where('customer_id', '=', $request->customer);
        }
        if (count(explode('to', $request->issue_date)) > 1) {
            $date_range = explode(' to ', $request->issue_date);
            $query->whereBetween('issue_date', $date_range);
        } elseif (!empty($request->issue_date)) {
            $date_range = [$request->issue_date, $request->issue_date];
            $query->whereBetween('issue_date', $date_range);
        }
        if ($request->status != null) {
            $query->where('status', '=', $request->status);
        }
        $ewaybills = $query->get();

        return view('ewaybill.index', compact('ewaybills', 'customer', 'status'));
        } else {
        return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }


    public function create($customerId = 0)
    {
        if (!\Auth::user()->can('create invoice')) {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    
        // Fetch necessary data
        $customFields = CustomField::where('created_by', '=', \Auth::user()->creatorId())
            ->where('module', '=', 'ewaybill')
            ->get();
        $ewaybill_number = \Auth::user()->ebillNumberFormat($this->ebillNumber());
        $customers = Customer::where('created_by', \Auth::user()->creatorId())
            ->get()
            ->pluck('name', 'id');
        $customers->prepend('Select Customer', '');
        $category = ProductServiceCategory::where('created_by', \Auth::user()->creatorId())
            ->where('type', 'income')
            ->get()
            ->pluck('name', 'id');
        $category->prepend('Select Category', '');
        $product_services = ProductService::where('created_by', \Auth::user()->creatorId())
            ->get()
            ->pluck('name', 'id');
        $product_services->prepend('--', '');
    
        $customer = $customerId ? Customer::find($customerId) : null;
        $settings_data = Utility::settingsById(\Auth::user()->creatorId());
        $company_state = $settings_data['company_state'] ?? '';
        $billing_state = $customer ? strtolower(trim($customer->billing_state ?? '')) : '';
    
        $tax_type = strtolower($settings_data['tax_type'] ?? 'gst');
        $deliveryDistance = request()->get('delivery_distance', 0);
    
       
        return view('ewaybill.create', compact(
            'customers',
            'ewaybill_number',
            'product_services',
            'category',
            'customFields',
            'customerId',
            'customer',
            'settings_data',
            'company_state',
            'billing_state',
            'deliveryDistance',
            'tax_type'
        ));
    }
    
    

    public function customer(Request $request)
    {
        $customer = Customer::where('id', '=', $request->id)->first();

        if ($customer) {
            $billingState = strtolower(trim($customer->billing_state ?? ''));
            // $settings = \App\Models\Setting::where('created_by', \Auth::user()->creatorId())->pluck('value', 'name')->toArray();
            $companyState = strtolower(trim($settings['company_state'] ?? ''));
            $customerDetailView = view('ewaybill.customer_detail', compact('customer'))->render();

            return response()->json([
                'view' => $customerDetailView,
                'state' => $billingState,
                'city' => $customer->billing_city,
                'country' => $customer->billing_country,
                'address' => $customer->billing_address,
                'postal_code' => $customer->billing_zip,
            ]);
        }

        return response()->json(['error' => 'Customer not found'], 404);
    }

    public function product(Request $request)
    {
        $data['product'] = $product = ProductService::find($request->product_id);
        if (!$product) {
           
            return json_encode(['error' => 'Product not found']);
        }
        $data['unit'] = (!empty($product->unit)) ? $product->unit->name : '';
        $data['tax_id'] = $product->tax_id ?? 0;
        $data['taxes'] = $product->tax_id ? $product->tax($product->tax_id) : [];
        $data['salePrice'] = floatval($product->sale_price);
        $data['quantity'] = 1;
        $data['discount'] = 0;
        $data['totalAmount'] = $data['salePrice'] - $data['discount'];

        return json_encode($data);
    }

    public function store(Request $request)
    {
          if (!\Auth::user()->can('create invoice')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
        try {
            // Validate the request
            $validator = \Validator::make(
                $request->all(),
                [
                'customer_id' => 'required|exists:customers,id',
                'issue_date' => 'required|date',
                'due_date' => 'required|date',
                'category_id' => 'required',
                'ref_number' => 'nullable|string|max:255',
                'delivery_distance' => 'required|numeric|min:0',
                'items' => 'required|array',
                'items.*.item' => 'required|exists:product_services,id',
                'items.*.quantity' => 'required|numeric|min:0',
                'items.*.price' => 'required|numeric|min:0',
                'items.*.discount' => 'required|numeric|min:0',
                'items.*.description' => 'nullable|string|max:1000',
            ]);

            
           if ($request->delivery_distance < 4) {
    return redirect()->back()
        ->withInput()
        ->withErrors(['delivery_distance' => 'E-Way Bill can only be generated if delivery distance is more than 4 km.']);
}



            if ($validator->fails()) {
                return redirect()->back()->with('error', $validator->errors()->first())->withInput();
            }

            // Start a database transaction
            DB::beginTransaction();
            $settings = Utility::settingsById(\Auth::user()->creatorId());
            $tax_type = strtolower($settings['tax_type'] ?? 'gst');
           
        $ewaybill = new EWayBill();
        $ewaybill->ewaybill_id = $this->ebillNumber();
        $ewaybill->customer_id = $request->customer_id;
        $ewaybill->issue_date = $request->issue_date;
        $ewaybill->due_date = $request->due_date;
        $ewaybill->category_id = $request->category_id;
        $ewaybill->ref_number = $request->ref_number;
        $ewaybill->delivery_distance = $request->delivery_distance;
        $ewaybill->created_by = \Auth::user()->creatorId();
        $ewaybill->save();


            // Save custom fields if present
            if ($request->has('custom_fields')) {
                $ewaybill->customFields()->sync($request->custom_fields);
            }

            // Save products
            foreach ($request->items as $item) {
                if (empty($item['item'])) {
                    continue;
                }

                $productService = ProductService::find($item['item']);
                if (!$productService) {
                    DB::rollBack();
                    return redirect()->back()->with('error', __('Product not found.'))->withInput();
                }

                $ewaybillProduct = new EWayBillProduct();
                $ewaybillProduct->ewaybill_id = $ewaybill->id;
                $ewaybillProduct->product_id = $item['item'];
                $ewaybillProduct->quantity = floatval($item['quantity'] ?? 0);
                $ewaybillProduct->price = floatval($item['price'] ?? 0);
                $ewaybillProduct->discount = floatval($item['discount'] ?? 0);
                $ewaybillProduct->description = trim($item['description'] ?? '');
                $ewaybillProduct->tax = $productService->tax_id ? implode(',', (array) $productService->tax_id) : '';
                $ewaybillProduct->save();

                
            }

            DB::commit();

            return redirect()->route('ewaybill.index')->with('success', __('E-Bill successfully created.'));
        } catch (\Exception $e) {
            
            return redirect()->back()->with('error', __('An error occurred. Please try again.'))->withInput();
        }
    }
    public function edit($ids)
    {
        if (!\Auth::user()->can('edit invoice')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }

        try {
            $id = Crypt::decrypt($ids);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Something went wrong.'));
        }

        $ewaybill = EWayBill::find($id);
        if (!$ewaybill) {
            return redirect()->back()->with('error', __('E-Way Bill not found.'));
        }

        $ewaybill_number = \Auth::user()->ebillNumberFormat($ewaybill->ewaybill_id);

        $creatorId = \Auth::user()->creatorId();
        $settings = DB::table('settings')->where('created_by', $creatorId)->pluck('value', 'name')->toArray();
        $settings_data = isset($ewaybill) ? \App\Models\Utility::settingsById($ewaybill->created_by) : \App\Models\Utility::settingsById($creatorId);
        $taxType = strtolower(trim($settings['tax_type'] ?? $settings_data['tax_type'] ?? 'gst'));
        $company_state = strtolower(trim($settings_data['company_state'] ?? 'defaultstate'));
      

        $customers = Customer::where('created_by', $creatorId)->pluck('name', 'id');
        $category = ProductServiceCategory::where('created_by', $creatorId)
            ->where('type', 'income')
            ->pluck('name', 'id')
            ->prepend('Select Category', '');
        $product_services = ProductService::where('created_by', $creatorId)->pluck('name', 'id');

        $ewaybill->customField = CustomField::getData($ewaybill, 'ewaybill');
        $customFields = CustomField::where('created_by', $creatorId)
            ->where('module', 'ewaybill')
            ->get();

        $items = [];
        $processedIds = [];

        foreach ($ewaybill->items as $billItem) {
            $billItem->itemAmount = $billItem->quantity * $billItem->price;
            $billItem->taxes = Utility::tax($billItem->tax) ?: [];
            $billItem->pro_description = $billItem->description ?? '';

            if ($taxType === 'vat') {
                $vatRate = 0;
                foreach ($billItem->taxes as $tax) {
                    $rate = floatval($tax->rate ?? 0);
                    if (in_array(strtoupper($tax->name ?? ''), ['CGST', 'SGST'])) {
                        $vatRate += $rate;
                    }
                }
                $billItem->itemTaxRate = $vatRate;
            } else {
                $billItem->itemTaxRate = $billItem->tax 
                    ? (json_decode($billItem->tax, true)[0]['rate'] ?? 0) 
                    : 0;
            }

          

            if (!in_array($billItem->product_id, $processedIds)) {
                $items[] = [
                    'id' => $billItem->id,
                    'product_id' => $billItem->product_id,
                    'quantity' => $billItem->quantity,
                    'price' => $billItem->price,
                    'discount' => $billItem->discount,
                    'description' => $billItem->pro_description,
                    'itemTaxRate' => $billItem->itemTaxRate,
                ];
                $processedIds[] = $billItem->product_id;
            }
        }

        $customer = Customer::find($ewaybill->customer_id);

        return view('ewaybill.edit', compact(
            'customers',
            'product_services',
            'ewaybill',
            'ewaybill_number',
            'category',
            'customFields',
            'items',
            'customer',
            'settings_data',
            'taxType',
            'company_state'
        ))->with('items_json', json_encode($items));
    }
    public function update(Request $request, EWayBill $ewaybill)
    {
        if (!\Auth::user()->can('edit invoice')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    
        if ($ewaybill->created_by != \Auth::user()->creatorId()) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    
        $validator = \Validator::make($request->all(), [
            'customer_id'         => 'required|exists:customers,id',
            'issue_date'          => 'required|date',
            'due_date'            => 'required|date',
            'category_id'         => 'required|exists:product_service_categories,id',
            'ref_number'          => 'nullable|string|max:255',
            'delivery_distance'   => 'required|numeric|min:0',
            'e_way_bill_number'   => 'nullable|string|max:255',
            'items'               => 'required|array',
            'items.*.item'        => 'required|exists:product_services,id',
            'items.*.quantity'    => 'required|numeric|min:0',
            'items.*.price'       => 'required|numeric|min:0',
            'items.*.discount'    => 'required|numeric|min:0',
            'items.*.description' => 'nullable|string|max:1000',
        ]);
    
        if ($request->delivery_distance < 4) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['delivery_distance' => 'E-Way Bill can only be generated if delivery distance is more than 4 km.']);
        }
    
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }
    
        DB::beginTransaction();
    
        try {
            $creatorId = \Auth::user()->creatorId();
            $settings_data = Utility::settingsById($creatorId);
            $taxType = strtoupper(trim($settings_data['tax_type'] ?? 'GST'));
    
            // Reverse previous customer balance
            Utility::updateUserBalance('customer', $ewaybill->customer_id, $ewaybill->getTotal(), 'debit');
    
            $ewaybill->fill([
                'customer_id'        => $request->customer_id,
                'issue_date'         => $request->issue_date,
                'due_date'           => $request->due_date,
                'ref_number'         => $request->ref_number,
                'e_way_bill_number'  => $request->delivery_distance > 4
                                        ? ($request->e_way_bill_number ?: $ewaybill->e_way_bill_number)
                                        : null,
                'delivery_distance'  => $request->delivery_distance,
                'category_id'        => $request->category_id,
            ])->save();
    
            if ($request->has('custom_fields')) {
                $ewaybill->customFields()->sync($request->custom_fields);
            }
    
            // Filter unique products
            $uniqueItems = collect($request->items)->unique('item')->values();
    
            foreach ($uniqueItems as $product) {
                $ewaybillProduct = EWayBillProduct::firstOrNew([
                    'ewaybill_id' => $ewaybill->id,
                    'product_id'  => $product['item'],
                ]);
    
                $productService = ProductService::find($product['item']);
                if (!$productService) {
                   
                    continue;
                }
    
                // VAT or GST tax array
                $taxArray = [];
                $taxId = $productService->tax_id ?? null;
    
                if (!empty($taxId)) {
                    $taxArray = is_array($taxId)
                        ? $taxId
                        : explode(',', $taxId);
                }
    
                $ewaybillProduct->fill([
                    'quantity'    => floatval($product['quantity']),
                    'discount'    => floatval($product['discount']),
                    'price'       => floatval($product['price']),
                    'description' => $product['description'] ?? '',
                    'tax'         => json_encode($taxArray), // store tax as IDs only
                ])->save();
    
                // OPTIONAL: Adjust stock if EWayBill affects inventory
                // Utility::total_quantity('minus', $ewaybillProduct->quantity, $ewaybillProduct->product_id);
            }
    
            // Update customer balance with new total
            Utility::updateUserBalance('customer', $ewaybill->customer_id, $ewaybill->getTotal(), 'credit');
    
            DB::commit();
    
            return redirect()->route('ewaybill.index')
                ->with('success', __('E-Way Bill successfully updated.'));
        } catch (\Exception $e) {
            DB::rollBack();
           
            return redirect()->back()->with('error', __('An error occurred. Please try again.'))->withInput();
        }
    }
    
public function ebillNumber()
{
    $creatorId = \Auth::user()->creatorId();
    
    $latest = EWayBill::where('created_by', '=', $creatorId)->latest()->first();
   
    return $latest ? $latest->ewaybill_id + 1 : 1;
}

public function ewaybillNumber()
{
    $creatorId = \Auth::user()->creatorId();

    $latest = EWayBill::where('created_by', $creatorId)->latest('id')->first();

    return $latest ? $latest->ewaybill_id + 1 : 1;
}

public function generateEwayBillNumber(Request $request)
{
    try {
        $deliveryDistance = $request->input('delivery_distance', 0);

        if ($deliveryDistance > 4) {
            $e_way_bill_number = \Auth::user()->ewaybillNumberFormat($this->ewaybillNumber());
            return response()->json(['e_way_bill_number' => $e_way_bill_number]);
        }

        return response()->json(['e_way_bill_number' => null]);
    } catch (\Throwable $e) {
       
        return response()->json(['e_way_bill_number' => null], 500);
    }
}



public function show($ids)
    {
        try {
            $id = Crypt::decrypt($ids);
            $ewaybill = EWayBill::with(['items.product.unit', 'payments', 'creditNote'])->find($id);
           
            // ... rest of the code
        } catch (\Throwable $th) {
           
            return redirect()->back()->with('error', __('EWayBill Not Found.'));
        }

        $ewaybill = EWayBill::with(['items.product.unit', 'payments', 'creditNote'])->find($id);

        if (!$ewaybill || $ewaybill->created_by != \Auth::user()->creatorId()) {
           
            return redirect()->back()->with('error', __('Permission denied.'));
        }

        $ewaybillPayment = $ewaybill->payments()->first();
        $customer = $ewaybill->customer;
        $items = $ewaybill->items;
        $user = \Auth::user();
        $ewaybill_user = User::find($ewaybill->created_by);
        $user_plan = Plan::getPlan($ewaybill_user->plan);

        $ewaybill->customField = CustomField::getData($ewaybill, 'ewaybill');
        $customFields = CustomField::where('created_by', \Auth::user()->creatorId())
            ->where('module', 'ewaybill')
            ->get();
        $creditnote = $ewaybill->creditNote()->first();

        // Determine state and tax logic
        // $settings = \App\Models\Setting::where('created_by', $ewaybill->created_by)->pluck('value', 'name')->toArray();
        $companyState = strtolower(trim($settings['company_state'] ?? ''));
        $billingState = strtolower(trim($customer->billing_state ?? ''));
        $isSameState = $companyState && $billingState && $companyState === $billingState;

        // Tax Calculation
        $taxesData = ['CGST' => 0, 'SGST' => 0, 'IGST' => 0];
        $itemData = [];

        foreach ($ewaybill->items as $invItem) {
            $product = $invItem->product;
            $item = new \stdClass();
            $item->name = $product->name ?? '';
            $item->quantity = $invItem->quantity;
            $item->price = $invItem->price;
            $item->discount = $invItem->discount;
            $item->unit = $product->unit_id ?? '';
            $item->description = $invItem->description;
            $item->itemTax = [];

            $baseAmount = ($item->price - $item->discount) * $item->quantity;

            $taxRates = Utility::tax($invItem->tax);

            foreach ($taxRates as $tax) {
                if (isset($tax->rate)) {
                    $rate = floatval($tax->rate);
                    $taxAmount = Utility::taxRate($rate, $item->price, $item->quantity, $item->discount);

                    if (isset($tax->type) && strtolower($tax->type) == 'vat') {
                        $item->itemTax[] = [
                            'name' => 'VAT',
                            'rate' => $rate . '%',
                            'tax_price' => $taxAmount,
                        ];
                        $taxesData['VAT'] = ($taxesData['VAT'] ?? 0) + $taxAmount;
                    }
                    else{

                    if($isSameState) {
                        $half = $taxAmount / 2;
                        $item->itemTax[] = ['name' => 'CGST', 'rate' => $rate . '%', 'tax_price' => $half];
                        $item->itemTax[] = ['name' => 'SGST', 'rate' => $rate . '%', 'tax_price' => $half];
                        $taxesData['CGST'] += $half;
                        $taxesData['SGST'] += $half;
                    } else {
                        $item->itemTax[] = ['name' => 'IGST', 'rate' => $rate . '%', 'tax_price' => $taxAmount];
                        $taxesData['IGST'] += $taxAmount;
                    }
                }
            }
        }
            $itemData[] = $item;
        }

        $ewaybill->itemData = $itemData;

        return view('ewaybill.view', compact(
            'ewaybill', 'customer', 'items', 'ewaybillPayment',
            'customFields', 'user', 'ewaybill_user', 'user_plan',
            'creditnote', 'taxesData', 'isSameState'
        ));
    }

    public function destroy(EWayBill $ewaybill, Request $request)
    {
        if (\Auth::user()->can('delete invoice')) {
            if ($ewaybill->created_by == \Auth::user()->creatorId()) {
                foreach ($ewaybill->payments as $payment) {
                    Utility::bankAccountBalance($payment->account_id, $payment->amount, 'debit');
                    $payment->delete();
                }

                if ($ewaybill->customer_id != 0 && $ewaybill->status != 0) {
                    Utility::updateUserBalance('customer', $ewaybill->customer_id, $ewaybill->getDue(), 'debit');
                }

                TransactionLines::where('reference_id', $ewaybill->id)->where('reference', 'ewaybill')->delete();
                TransactionLines::where('reference_id', $ewaybill->id)->where('reference', 'ewaybill Payment')->delete();
                CreditNote::where('ewaybill', '=', $ewaybill->id)->delete();
                EwayBillProduct::where('ewaybill_id', '=', $ewaybill->id)->delete();
                $ewaybill->delete();

                return redirect()->route('ewaybill.index')->with('success', __('ewaybill successfully deleted.'));
            } else {
                return redirect()->back()->with('error', __('Permission denied.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function productDestroy(Request $request)
    {
        if (\Auth::user()->can('delete invoice product')) {
            $ewaybillProduct = EWayBillProduct::find($request->id);

            if ($ewaybillProduct) {
                $ewaybill = EWayBill::find($ewaybillProduct->ewaybill_id);
                $productService = ProductService::find($ewaybillProduct->product_id);

                Utility::updateUserBalance('customer', $ewaybill->customer_id, $request->amount, 'debit');
                TransactionLines::where('reference_sub_id', $productService->id)->where('reference', 'ewaybill')->delete();
                $ewaybillProduct->delete();

                return response()->json(['status' => true, 'message' => __('ewaybill product successfully deleted.')]);
            } else {
                return response()->json(['status' => false, 'message' => __('ewaybill product not found!')]);
            }
        } else {
            return response()->json(['status' => false, 'message' => __('Permission denied.')]);
        }
    }

    public function customerewaybill(Request $request)
    {
        if (\Auth::user()->can('manage customer invoice')) {
            $status = EWayBill::$statues;
            $query = EWayBill::where('customer_id', '=', \Auth::user()->id)->where('status', '!=', '0')->where('created_by', \Auth::user()->creatorId());

            if (!empty($request->issue_date)) {
                $date_range = explode(' - ', $request->issue_date);
                $query->whereBetween('issue_date', $date_range);
            }

            if (!empty($request->status)) {
                $query->where('status', '=', $request->status);
            }
            $ewaybills = $query->get();

            return view('ewaybill.index', compact('ewaybills', 'status'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }

    public function customerewaybillShow($id)
    {
        if (\Auth::user()->can('show invoice')) {
            try {
                $ewaybill = EWayBill::with('payments.bankAccount', 'items.product.unit')->find($id);

                if (!$ewaybill) {
                  
                    return redirect()->back()->with('error', __('ewaybill not found.'));
                }

                $user = User::where('id', $ewaybill->created_by)->first();

                if ($ewaybill->created_by != $user->creatorId()) {
                    
                    return redirect()->back()->with('error', __('Permission denied.'));
                }

                $customer = $ewaybill->customer;
                if (!$customer) {
                    Log::warning('Customer not found', ['customer_id' => $ewaybill->customer_id]);
                }

                $iteams = $ewaybill->items ?? collect([]);

                $rawItems = \DB::table('ewaybill_products')
                    ->where('ewaybill_id', $id)
                    ->get()
                    ->toArray();
               ;

                $settings = method_exists($user, 'creatorId')
                    ? DB::table('settings')->where('created_by', $user->creatorId())->pluck('value', 'name')->toArray()
                    : [];

                $companySettings = Utility::settingsById($ewaybill->created_by);
                $customFields = \App\Models\CustomField::where('created_by', $ewaybill->created_by)->get();

                $companyState = strtolower(trim($settings['company_state'] ?? ''));
                $billingState = strtolower(trim($customer->billing_state ?? ''));
                $isSameState = $companyState && $billingState && $companyState === $billingState;

              

                $view = $user->type == 'company' ? 'ewaybill.customer_ewaybill' : 'ewaybill.view';
                
                return view($view, compact('ewaybill', 'customer', 'iteams', 'user', 'settings', 'companySettings', 'customFields', 'isSameState'));
            } catch (\Exception $e) {
               
                return redirect()->back()->with('error', __('An error occurred while processing the ewaybill. Please try again.'));
            }
        } else {
     
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }


     public function sent($id)
    {
        if (\Auth::user()->can('send invoice')) {
            $setings = Utility::settings();

            if ($setings['customer_invoice_sent'] == 1) {
                $ewaybill = EWayBill::where('id', $id)->first();
                $ewaybill->send_date = date('Y-m-d');
                $ewaybill->status = 1;
                $ewaybill->save();

                $customer = Customer::where('id', $ewaybill->customer_id)->first();
                $ewaybill->name = !empty($customer) ? $customer->name : '';
                $ewaybill->ewaybill = \Auth::user()->ebillNumberFormat($ewaybill->ewaybill_id);
                $ewaybillId = Crypt::encrypt($ewaybill->id);
                $ewaybill->url = route('ewaybill.pdf', $ewaybillId);

                $ewaybill_products = EWayBillProduct::where('ewaybill_id', $ewaybill->id)->get();
                foreach ($ewaybill_products as $ewaybill_product) {
                    $product = ProductService::find($ewaybill_product->product_id);
                    $totalTaxPrice = 0;

                    if ($ewaybill_product->tax != null) {
                        $taxPrice = Utility::taxRate($ewaybill_product->tax, $ewaybill_product->price, $ewaybill_product->quantity, $ewaybill_product->discount);
                        $totalTaxPrice += $taxPrice;
                    }

                    $itemAmount = ($ewaybill_product->price * $ewaybill_product->quantity) - ($ewaybill_product->discount) + $totalTaxPrice;

                    $data = [
                        'account_id' => $product->sale_chartaccount_id,
                        'transaction_type' => 'Credit',
                        'transaction_amount' => $itemAmount,
                        'reference' => 'EWayBill',
                        'reference_id' => $ewaybill->id,
                        'reference_sub_id' => $product->id,
                        'date' => $ewaybill->issue_date,
                    ];
                    Utility::addTransactionLines($data, 'create');
                }

                $customerArr = [
                    'customer_name' => $customer->name,
                    'customer_email' => $customer->email,
                    'invoice_name' => $customer->name,
                    'invoice_number' => $ewaybill->ewaybill,
                    'invoice_url' => $ewaybill->url,
                ];
                $resp = Utility::sendEmailTemplate('customer_invoice_sent', [$customer->id => $customer->email], $customerArr);
               
                if (!empty($customer->contact)) {
            
                    $number = preg_replace('/[^0-9]/', '', $customer->contact);
                    $message = "Dear {$customer->name}, your ewaybill ({$ewaybill->ewaybill}) is ready. Please check your email.";
                    $whatsappStatus = Utility::sendWhatsappMessage($number, $message);
            
                } 


                return redirect()->back()->with('success', __('ewaybill successfully sent.') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
            } else {
                return redirect()->back()->with('error', 'ewaybill not found');
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function resent($id)
    {
        if (\Auth::user()->can('send invoice')) {
            $ewaybill = EWayBill::where('id', $id)->first();
            $customer = Customer::where('id', $ewaybill->customer_id)->first();
            $ewaybill->name = !empty($customer) ? $customer->name : '';
            $ewaybill->ewaybill = \Auth::user()->ebillNumberFormat($ewaybill->ewaybill_id);
            $ewaybillId = Crypt::encrypt($ewaybill->id);
            $ewaybill->url = route('ewaybill.pdf', $ewaybillId);

            $customerArr = [
                'customer_name' => $customer->name,
                'customer_email' => $customer->email,
                'invoice_name' => $customer->name,
                'invoice_number' => $ewaybill->ewaybill,
                'invoice_url' => $ewaybill->url,
            ];
            $resp = Utility::sendEmailTemplate('customer_invoice_sent', [$customer->id => $customer->email], $customerArr);
            if (!empty($customer->contact)) {
            
                $number = preg_replace('/[^0-9]/', '', $customer->contact);
                $message = "Dear {$customer->name}, your ewaybill ({$ewaybill->ewaybill}) is ready. Please check your email.";
                $whatsappStatus = Utility::sendWhatsappMessage($number, $message);
        
            } 

            return redirect()->back()->with('success', __('ewaybill successfully sent.') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }


    public function customerewaybillSend($ewaybill_id)
    {
        return view('customer.ewaybill_send', compact('ewaybill_id'));
    }

    public function customerewaybillSendMail(Request $request, $ewaybill_id)
    {
        $validator = \Validator::make(
            $request->all(),
            ['email' => 'required|email']
        );
        if ($validator->fails()) {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error', $messages->first());
        }

        $email = $request->email;
        $ewaybill = EWayBill::where('id', $ewaybill_id)->first();
        $customer = Customer::where('id', $ewaybill->customer_id)->first();
        $ewaybill->name = !empty($customer) ? $customer->name : '';
        // $settings = \App\Models\Setting::where('created_by', \Auth::user()->creatorId())->pluck('value', 'name')->toArray();
        $ewaybill->ewaybill =\Auth::user()->ebillNumberFormat($ewaybill->ewaybill_id);
        $ewaybillId = Crypt::encrypt($ewaybill->id);
        $ewaybill->url = route('ewaybill.pdf', $ewaybillId);

        try {
            Mail::to($email)->send(new CustomerewaybillSend($ewaybill));
        } catch (\Exception $e) {
            $smtp_error = __('E-Mail has been not sent due to SMTP configuration');
        }

        return redirect()->back()->with('success', __('ewaybill successfully sent.') . ((isset($smtp_error)) ? '<br> <span class="text-danger">' . $smtp_error . '</span>' : ''));
    }

    public function shippingDisplay(Request $request, $id)
    {
        $ewaybill = EWayBill::find($id);
        $ewaybill->shipping_display = $request->is_display == 'true' ? 1 : 0;
        $ewaybill->save();

        return redirect()->back()->with('success', __('Shipping address status successfully changed.'));
    }

    public function duplicate($ewaybill_id)
    {
        if (\Auth::user()->can('duplicate invoice')) {
            $ewaybill =EWaybill::where('id', $ewaybill_id)->first();
            $duplicateewaybill = new EWayBill();
            $duplicateewaybill->ewaybill_id = $this->ewaybillNumber();
            $duplicateewaybill->customer_id = $ewaybill->customer_id;
            $duplicateewaybill->issue_date = date('Y-m-d');
            $duplicateewaybill->due_date = $ewaybill->due_date;
            $duplicateewaybill->send_date = null;
            $duplicateewaybill->category_id = $ewaybill->category_id;
            $duplicateewaybill->ref_number = $ewaybill->ref_number;
            $duplicateewaybill->status = 0;
            $duplicateewaybill->shipping_display = $ewaybill->shipping_display;
            $duplicateewaybill->created_by = $ewaybill->created_by;
            $duplicateewaybill->save();

            $ewaybillProduct = EWayBillProduct::where('ewaybill_id', $ewaybill_id)->get();
            foreach ($ewaybillProduct as $product) {
                $duplicateProduct = new EWayBillProduct();
                $duplicateProduct->ewaybill_id = $duplicateewaybill->id;
                $duplicateProduct->product_id = $product->product_id;
                $duplicateProduct->quantity = $product->quantity;
                $duplicateProduct->tax = $product->tax;
                $duplicateProduct->discount = $product->discount;
                $duplicateProduct->price = $product->price;
                $duplicateProduct->description = $product->description;
                $duplicateProduct->save();
            }

            return redirect()->back()->with('success', __('ewaybill duplicate successfully.'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

public function previewewaybill(Request $request, $template, $color)
    {
        $objUser = \Auth::user();
        $settings = Utility::settings();
        $ewaybill = new EWayBill();

        $customer = new \stdClass();
        $customer->email = '<Email>';
        $customer->shipping_name = '<Customer Name>';
        $customer->shipping_country = '<Country>';
        $customer->shipping_state = '<State>';
        $customer->shipping_city = '<City>';
        $customer->shipping_phone = '<Customer Phone Number>';
        $customer->shipping_zip = '<Zip>';
        $customer->shipping_address = '<Address>';
        $customer->billing_name = '<Customer Name>';
        $customer->billing_country = '<Country>';
        $customer->billing_state = '<State>';
        $customer->billing_city = '<City>';
        $customer->billing_phone = '<Customer Phone Number>';
        $customer->billing_zip = '<Zip>';
        $customer->billing_address = '<Address>';

        $totalTaxPrice = 0;
        $taxesData = [];
        $items = [];
        for ($i = 1; $i <= 3; $i++) {
            $item = new \stdClass();
            $item->name = 'Item ' . $i;
            $item->quantity = 1;
            $item->tax = 5;
            $item->discount = 50;
            $item->price = 100;
            $item->unit = 1;
            $item->description = 'XYZ';

            $taxes = [
                'Tax 1',
                'Tax 2',
            ];

            $itemTaxes = [];
            foreach ($taxes as $k => $tax) {
                $taxPrice = 10;
                $totalTaxPrice += $taxPrice;
                $itemTax['name'] = 'Tax ' . $k;
                $itemTax['rate'] = '10 %';
                $itemTax['price'] = '$10';
                $itemTax['tax_price'] = 10;
                $itemTaxes[] = $itemTax;
                $taxesData['Tax ' . $k] = ($taxesData['Tax ' . $k] ?? 0) + $taxPrice;
            }
            $item->itemTax = $itemTaxes;
            $items[] = $item;
        }

        $ewaybill->ewaybill_id = 1;
        $ewaybill->issue_date = date('Y-m-d H:i:s');
        $ewaybill->due_date = date('Y-m-d H:i:s');
        $ewaybill->itemData = $items;
        $ewaybill->status = 0;
        $ewaybill->totalTaxPrice = 60;
        $ewaybill->totalQuantity = 3;
        $ewaybill->totalRate = 300;
        $ewaybill->totalDiscount = 10;
        $ewaybill->taxesData = $taxesData;
        $ewaybill->created_by = $objUser->creatorId();
        $accounts = BankAccount::where('created_by', $objUser->creatorId())->get();

            $invoicePayment = \App\Models\InvoicePayment::with('bankAccount')
    ->where('ewaybill_id', $ewaybill->id)
    ->latest()
    ->first(); 


        $ewaybill->customField = [];
        $customFields = [];

        $preview = 1;
        $color = '#' . $color;
        $fontColor = Utility::getFontColor($color);

 $logo = asset(Storage::url('uploads/logo/'));
        $company_logo = Utility::getValByName('company_logo_dark');
        
        $ewaybill_logo = Utility::getValByName('ewaybill_logo');
     $img = isset($ewaybill_logo) && !empty($ewaybill_logo)
    ? asset('storage/ewaybill_logo/' . $ewaybill_logo)
    : asset($logo . '/' . (isset($company_logo) && !empty($company_logo) ? $company_logo : 'logo-dark.png'));

        return view('ewaybill.templates.' . $template, compact('ewaybill', 'preview', 'color', 'img', 'settings', 'customer', 'fontColor', 'customFields', 'accounts','invoicePayment',));
    }

    public function ewaybill($ewaybill_id)
    {
        try {
            $settings = Utility::settings();
            $ewaybillId = Crypt::decrypt($ewaybill_id);
            $ewaybill = EWayBill::with('items.product')->find($ewaybillId);

            if (!$ewaybill) {
               
                return redirect()->back()->with('error', __('ewaybill not found.'));
            }

            $creatorSettings = DB::table('settings')->where('created_by', $ewaybill->created_by)->pluck('value', 'name')->toArray();
            $settings = array_merge($settings, $creatorSettings);

            $customer = $ewaybill->customer;
            if (!$customer) {
                Log::warning('Customer not found', ['customer_id' => $ewaybill->customer_id]);
            }

            $companyState = strtolower(trim($creatorSettings['company_state'] ?? ''));
            $billingState = strtolower(trim($customer->billing_state ?? ''));
            $isSameState = $companyState && $billingState && $companyState === $billingState;

            $items = [];
            $taxesData = [
                'CGST' => 0,
                'SGST' => 0,
                'IGST' => 0,
                'VAT'  => 0,
            ];
            $totalTax = $totalQuantity = $totalRate = $totalDiscount = 0;

            foreach ($ewaybill->items as $invItem) {
                $item = new \stdClass();
                $product = $invItem->product;

                $item->name = $product->name ?? '';
                $item->quantity = $invItem->quantity;
                $item->unit = $product->unit_id ?? '';
                $item->price = $invItem->price;
                $item->discount = $invItem->discount;
                $item->description = $invItem->description;
                $item->expense_chartaccount_code = $product->expenseChartAccount->code ?? null;

                $totalQuantity += $item->quantity;
                $totalRate += $item->quantity * $item->price;
                $totalDiscount += $item->discount;

                $itemTaxes = [];
                $itemVATTotal = 0;

                if ($product && $invItem->tax) {
                    $taxRates = Utility::tax($invItem->tax);
                    $baseTaxAmount = 0;

                   

                    if (!empty($taxRates)) {
                        foreach ($taxRates as $tax) {
                            if ($tax && isset($tax->rate)) {
                                $rate = floatval($tax->rate);
                                $taxAmount = Utility::taxRate($rate, $item->price, $item->quantity, $item->discount);
                                $baseTaxAmount += $taxAmount;

                                if (!empty($settings['tax_type']) && $settings['tax_type'] == 'VAT') {
                                    $taxesData['VAT'] += $taxAmount;
                                    $totalTax += $taxAmount;
                                
                                    $itemTaxes[] = [
                                        'name' => 'VAT',
                                        'rate' => ($rate * 2) . '%',
                                        'price' => Utility::priceFormat($settings, $taxAmount),
                                        'tax_price' => $taxAmount,
                                    ];
                                
                                 } elseif ($isSameState) {
                                    $half = $taxAmount / 2;
                                    $itemTaxes[] = [
                                        'name' => 'CGST',
                                        'rate' => $rate . '%',
                                        'price' => Utility::priceFormat($settings, $half),
                                        'tax_price' => $half,
                                    ];
                                    $itemTaxes[] = [
                                        'name' => 'SGST',
                                        'rate' => $rate . '%',
                                        'price' => Utility::priceFormat($settings, $half),
                                        'tax_price' => $half,
                                    ];
                                    $taxesData['CGST'] += $half;
                                    $taxesData['SGST'] += $half;
                                } else {
                                    $itemTaxes[] = [
                                        'name' => 'IGST',
                                        'rate' => $rate . '%',
                                        'price' => Utility::priceFormat($settings, $taxAmount),
                                        'tax_price' => $taxAmount,
                                    ];
                                    $taxesData['IGST'] += $taxAmount;
                                }
                            } else {
                                Log::warning('Invalid tax object detected.', ['tax' => $tax]);
                            }
                        }
                        // Apply VAT per item only once
                    if ($itemVATTotal > 0) {
                        $taxesData['VAT'] += $itemVATTotal;

                    }
                    }
                }

                $item->itemTax = $itemTaxes;
                $items[] = $item;
            }

            $ewaybill->itemData = $items;
            $ewaybill->taxesData = array_filter($taxesData, fn($value) => $value > 0);
            $ewaybill->totalTaxPrice = $totalTax;
            $ewaybill->totalRate = $totalRate;
            $ewaybill->totalDiscount = $totalDiscount;
            $ewaybill->customField = CustomField::getData($ewaybill, 'ewaybill');
    
            $customFields = auth()->check()
                ? CustomField::where('created_by', auth()->user()->creatorId())->where('module', 'ewaybill')->get()
                : [];
    

            $accounts = BankAccount::where('created_by', $ewaybill->created_by)->get();

            $invoicePayment = \App\Models\InvoicePayment::with('bankAccount')
    ->where('ewaybill_id', $ewaybill->id)
    ->latest()
    ->first(); 

            $logoPath = asset(Storage::url('uploads/logo/'));
            $ewaybillLogo = Utility::settingsById($ewaybill->created_by)['ewaybill_logo'] ?? '';
            $companyLogo = Utility::getValByName('company_logo_dark');
            $img = $ewaybillLogo
                ? Utility::get_file('ewaybill_logo/') . $ewaybillLogo
                : $logoPath . '/' . ($companyLogo ?: 'logo-dark.png');

            $color = '#' . ($settings['ewaybill_color'] ?? '000000');
            $fontColor = Utility::getFontColor($color);

            return view('ewaybill.templates.' . ($settings['ewaybill_template'] ?? 'default'), compact(
                'ewaybill', 'settings', 'customer', 'img', 'color', 'fontColor', 'customFields', 'accounts','invoicePayment'
            ));
        } catch (\Exception $e) {
           
            return redirect()->back()->with('error', __('An error occurred while generating ewaybill.'));
        }
    }

    public function saveEwaybillTemplateSettings(Request $request)
    {
        $post = $request->all();
        unset($post['_token']);

        if (isset($post['ewaybill_template']) && (!isset($post['ewaybill_color']) || empty($post['ewaybill_color']))) {
            $post['ewaybill_color'] = "ffffff";
        }

        if ($request->hasFile('ewaybill_logo')) {
            $dir = 'ewaybill_logo/';
            $ewaybill_logo = \Auth::user()->id . '_ewaybill_logo.png';
            $validation = [
                'mimes:png',
                'max:20480',
            ];
            $path = Utility::upload_file($request, 'ewaybill_logo', $ewaybill_logo, $dir, $validation);

            if ($path['flag'] == 0) {
                return redirect()->back()->with('error', __($path['msg']));
            }
            $post['ewaybill_logo'] = $ewaybill_logo;
        }

        foreach ($post as $key => $data) {
            \DB::insert(
                'insert into settings (`value`, `name`, `created_by`) values (?, ?, ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)',
                [$data, $key, \Auth::user()->creatorId()]
            );
        }
    
        return redirect()->back()->with('success', __('Eaybill Setting updated successfully'));
    }

        public function items(Request $request)
        {
            $items = EWayBillProduct::where('ewaybill_id', $request->ewaybill_id)->where('product_id', $request->product_id)->first();
            return json_encode($items);
        }
    
        public function ewaybillLink($ewaybillId)
        {
            try {
                $id = Crypt::decrypt($ewaybillId);
            } catch (\Throwable $th) {
                return redirect()->back()->with('error', __('ewaybill Not Found.'));
            }
    
            $ewaybill = EWayBill::with(['creditNote', 'payments.bankAccount', 'items.product.unit'])->find($id);
            
            $settings = Utility::settingsById($ewaybill->created_by);
    
            if (!empty($ewaybill)) {
                $user_id = $ewaybill->created_by;
                $user = User::find($user_id);
                $ewaybillPayment = InvoicePayment::where('ewaybill_id', $ewaybill->id)->get();
                $customer = $ewaybill->customer;
                $iteams = $ewaybill->items;
                $ewaybill->customField = CustomField::getData($ewaybill, 'ewaybill');
                $customFields = CustomField::where('module', '=', 'ewaybill')->where('created_by', $ewaybill->created_by)->get();
                $company_payment_setting = Utility::getCompanyPaymentSetting($user_id);
                $user_plan = Plan::find($user->plan);
    
                return view('ewaybill.customer_ewaybill', compact('settings', 'ewaybill', 'customer', 'iteams', 'ewaybillPayment', 'customFields', 'user', 'company_payment_setting', 'user_plan'));
            } else {
                return redirect()->back()->with('error', __('Permission Denied.'));
            }
        }

    public function export()
{
    $name = 'ewaybill_' . date('Y-m-d_H-i-s'); // corrected date format
    return Excel::download(new EWayBillExport(), $name . '.xlsx');
}

        public function payment($ewaybill_id)
        {
            if (\Auth::user()->can('create payment invoice')) {
                $ewaybill = EWaybill::where('id', $ewaybill_id)->first();
                $customers = Customer::where('created_by', '=', \Auth::user()->creatorId())->get()->pluck('name', 'id');
                $categories = ProductServiceCategory::where('created_by', '=', \Auth::user()->creatorId())->get()->pluck('name', 'id');
                $accounts = BankAccount::select('*', \DB::raw("CONCAT(bank_name,' ',holder_name) AS name"))->where('created_by', \Auth::user()->creatorId())->get()->pluck('name', 'id');
    
                return view('ewaybill.payment', compact('customers', 'categories', 'accounts', 'ewaybill'));
            } else {
                return redirect()->back()->with('error', __('Permission denied.'));
            }
        }
    
        public function createPayment(Request $request, $ewaybill_id)
        {
            $ewaybill = EWayBill::find($ewaybill_id);
            if ($ewaybill->getDue() < $request->amount) {
                return redirect()->back()->with('error', __('ewaybill payment amount should not greater than subtotal.'));
            }
    
            if (\Auth::user()->can('create payment invoice')) {
                $validator = \Validator::make(
                    $request->all(),
                    [
                        'date' => 'required',
                        'amount' => 'required|numeric|min:0',
                        'account_id' => 'required',
                    ]
                );
                if ($validator->fails()) {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }
    
                $ewaybillPayment = new InvoicePayment();
                $ewaybillPayment->ewaybill_id = $ewaybill_id;
                $ewaybillPayment->date = $request->date;
                $ewaybillPayment->amount = floatval($request->amount);
                $ewaybillPayment->account_id = $request->account_id;
                $ewaybillPayment->payment_method = 0;
                $ewaybillPayment->reference = $request->reference;
                $ewaybillPayment->description = $request->description;
    
                if (!empty($request->add_receipt)) {
                    $image_size = $request->file('add_receipt')->getSize();
                    $result = Utility::updateStorageLimit(\Auth::user()->creatorId(), $image_size);
                    if ($result == 1) {
                        $fileName = time() . "_" . $request->add_receipt->getClientOriginalName();
                        $request->add_receipt->storeAs('uploads/payment', $fileName);
                        $ewaybillPayment->add_receipt = $fileName;
                    }
                }
    
                $ewaybillPayment->save();
    
                $ewaybill = EWayBill::where('id', $ewaybill_id)->first();
                $due = $ewaybill->getDue();
                $total = $ewaybill->getTotal();
                if ($ewaybill->status == 0) {
                    $ewaybill->send_date = date('Y-m-d');
                    $ewaybill->save();
                }
    
                if ($due <= 0) {
                    $ewaybill->status = 4;
                    $ewaybill->save();
                } else {
                    $ewaybill->status = 3;
                    $ewaybill->save();
                }
    
                $ewaybillPayment->user_id = $ewaybill->customer_id;
                $ewaybillPayment->user_type = 'Customer';
                $ewaybillPayment->type = 'Partial';
                $ewaybillPayment->created_by = \Auth::user()->id;
                $ewaybillPayment->payment_id = $ewaybillPayment->id;
                $ewaybillPayment->category = 'ewaybill';
                $ewaybillPayment->account = $request->account_id;
    
                Transaction::addTransaction($ewaybillPayment);
                $customer = Customer::where('id', $ewaybill->customer_id)->first();
    
                $payment = new InvoicePayment();
                $payment->name = $customer['name'];
                $payment->date = \Auth::user()->dateFormat($request->date);
                $payment->amount = \Auth::user()->priceFormat($request->amount);
                $payment->ewaybill = 'ewaybill ' . \Auth::user()->ebillNumberFormat($ewaybill->ewaybill_id);
                $payment->dueAmount = \Auth::user()->priceFormat($ewaybill->getDue());
    
                Utility::updateUserBalance('customer', $ewaybill->customer_id, $request->amount, 'credit');
                Utility::bankAccountBalance($request->account_id, $request->amount, 'credit');
    
                $ewaybillPayments = InvoicePayment::where('ewaybill_id', $ewaybill->id)->get();
                foreach ($ewaybillPayments as $ewaybillPayment) {
                    $accountId = BankAccount::find($ewaybillPayment->account_id);
                    $data = [
                        'account_id' => $accountId->chart_account_id,
                        'transaction_type' => 'Debit',
                        'transaction_amount' => $ewaybillPayment->amount,
                        'reference' => 'ewaybill Payment',
                        'reference_id' => $ewaybill->id,
                        'reference_sub_id' => $ewaybillPayment->id,
                        'date' => $ewaybillPayment->date,
                    ];
                    Utility::addTransactionLines($data, 'create');
                }
    
                $setings = Utility::settings();
                if ($setings['new_invoice_payment'] == 1) {
                    $customer = Customer::where('id', $ewaybill->customer_id)->first();
                    $ewaybill_payment_method = InvoicePayment::where('ewaybill_id', $ewaybill->id)->orderBy('id', 'desc')->first();
                    $ewaybillPaymentArr = [
                        'ewaybill_payment_name' => $customer->name,
                        'ewaybill_payment_amount' => $payment->amount,
                        'ewaybill_payment_date' => $payment->date,
                        'payment_dueAmount' => $payment->dueAmount,
                        'ewaybill_number' => \Auth::user()->priceFormat($ewaybill->getDue()),
                        'ewaybill_payment_method' => $ewaybill_payment_method->payment_type,
                    ];
                    $resp = Utility::sendEmailTemplate('new_invoice_payment', [$customer->id => $customer->email], $ewaybillPaymentArr);

                    if (!empty($customer->contact)) {
                    $number = preg_replace('/[^0-9]/', '', $customer->contact);
                
                    $amount = floatval($request->amount);
                    $due = $ewaybill->getDue(); // This returns numeric value
                
                    $message = "Dear {$customer->name},\n"
                        . "Thank you for your payment of " . \Auth::user()->priceFormat($amount) . " on " . date('d-m-Y', strtotime($request->date)) . ".\n"
                        . "EWayBill Number: {$ewaybill->ewaybill_id}\n"
                        . "Payment Method: {$ewaybill_payment_method->payment_type}\n"
                        . "Remaining Due: " . \Auth::user()->priceFormat($due);
                
                    $whatsappStatus = Utility::sendWhatsappMessage($number, $message);
                }
                }
    
                $module = 'New ewaybill Payment';
                $webhook = Utility::webhookSetting($module);
                if ($webhook) {
                    $parameter = json_encode($ewaybill);
                    $status = Utility::WebhookCall($webhook['url'], $parameter, $webhook['method']);
                    if ($status == true) {
                        return redirect()->back()->with('success', __('Payment successfully added.') . ((isset($result) && $result != 1) ? '<br> <span class="text-danger">' . $result . '</span>' : '') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
                    } else {
                        return redirect()->back()->with('error', __('Payment successfully, Webhook call failed.'));
                    }
                }
                return redirect()->back()->with('success', __('Payment successfully added.') . ((isset($result) && $result != 1) ? '<br> <span class="text-danger">' . $result . '</span>' : '') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
            }
        }
    
        public function paymentDestroy(Request $request, $ewaybill_id, $payment_id)
        {
            if (\Auth::user()->can('delete payment invoice')) {
                $payment = InvoicePayment::find($payment_id);
                InvoicePayment::where('id', '=', $payment_id)->delete();
                InvoiceBankTransfer::where('id', '=', $payment_id)->delete();
                TransactionLines::where('reference_sub_id', $payment_id)->where('reference', 'ewaybill Payment')->delete();
    
                $ewaybill = EWayBill::where('id', $ewaybill_id)->first();
                $due = $ewaybill->getDue();
                $total = $ewaybill->getTotal();
    
                if ($due > 0 && $total != $due) {
                    $ewaybill->status = 3;
                } else {
                    $ewaybill->status = 2;
                }
    
                if (!empty($payment->add_receipt)) {
                    $file_path = '/uploads/payment/' . $payment->add_receipt;
                    Utility::changeStorageLimit(\Auth::user()->creatorId(), $file_path);
                }
    
                $ewaybill->save();
                $type = 'Partial';
                $user = 'Customer';
                Transaction::destroyTransaction($payment_id, $type, $user);
    
                Utility::updateUserBalance('customer', $ewaybill->customer_id, $payment->amount, 'credit');
                Utility::bankAccountBalance($payment->account_id, $payment->amount, 'debit');
    
                return redirect()->back()->with('success', __('Payment successfully deleted.'));
            } else {
                return redirect()->back()->with('error', __('Permission denied.'));
            }
        }
    
        public function paymentReminder($ewaybill_id)
        {
            $ewaybill = ewaybill::find($ewaybill_id);
            $customer = Customer::where('id', $ewaybill->customer_id)->first();
            $ewaybill->dueAmount = \Auth::user()->priceFormat($ewaybill->getDue());
            $ewaybill->name = $customer['name'];
            $ewaybill->date = \Auth::user()->dateFormat($ewaybill->send_date);
            $ewaybill->ewaybill = \Auth::user()->ebillNumberFormat($ewaybill->ewaybill_id);
    
            $setting = Utility::settings(\Auth::user()->creatorId());
            $reminderNotificationArr = [
                'ewaybill_number' => \Auth::user()->ebillNumberFormat($ewaybill->ewaybill_id),
                'customer_name' => $customer->name,
                'user_name' => \Auth::user()->name,
            ];
    
            if (isset($setting['twilio_reminder_notification']) && $setting['twilio_reminder_notification'] == 1) {
                Utility::send_twilio_msg($customer->contact, 'ewaybill_payment_reminder', $reminderNotificationArr);
            }
    
            $setings = Utility::settings();
            if ($setings['new_payment_reminder'] == 1) {
                $reminderArr = [
                    'payment_reminder_name' => $ewaybill->name,
                    'ewaybill_payment_number' => $ewaybill->ewaybill,
                    'ewaybill_payment_dueAmount' => $ewaybill->dueAmount,
                    'payment_reminder_date' => $ewaybill->date,
                ];
                $resp = Utility::sendEmailTemplate('new_payment_reminder', [$customer->id => $customer->email], $reminderArr);
                
                if (!empty($customer->contact)) {
                $number = preg_replace('/[^0-9]/', '', $customer->contact);
            
                $dueAmount = \Auth::user()->priceFormat($ewaybill->getDue());
                $dueDate = date('d-m-Y', strtotime($ewaybill->due_date ?? $ewaybill->date)); // use `due_date` if available
            
                $message = "Dear {$customer->name},\n"
                    . "This is a gentle reminder that payment for EWayBill #{$ewaybill->ewaybill_id} is due.\n"
                    . "Due Amount: ₹{$dueAmount}\n"
                    . "Due Date: {$dueDate}";
            
                $whatsappStatus = Utility::sendWhatsappMessage($number, $message);
            }
            }
    
            return redirect()->back()->with('success', __('Payment reminder successfully send.') . (($resp['is_success'] == false && !empty($resp['error'])) ? '<br> <span class="text-danger">' . $resp['error'] . '</span>' : ''));
        }
    
    }